/*
 * Decompiled with CFR 0.152.
 */
package com.floragunn.signals.truststore.service;

import com.floragunn.codova.validation.ConfigValidationException;
import com.floragunn.signals.CertificatesParser;
import com.floragunn.signals.truststore.service.CannotCreateTrustManagerException;
import com.floragunn.signals.truststore.service.TruststoreCrudService;
import com.floragunn.signals.truststore.service.persistence.TruststoreData;
import java.security.KeyStore;
import java.security.KeyStoreException;
import java.security.NoSuchAlgorithmException;
import java.security.cert.Certificate;
import java.util.Arrays;
import java.util.Collection;
import java.util.Collections;
import java.util.HashMap;
import java.util.HashSet;
import java.util.List;
import java.util.Map;
import java.util.Objects;
import java.util.Optional;
import java.util.stream.Collectors;
import javax.net.ssl.TrustManager;
import javax.net.ssl.TrustManagerFactory;
import javax.net.ssl.X509ExtendedTrustManager;
import org.apache.logging.log4j.LogManager;
import org.apache.logging.log4j.Logger;

public class TrustManagerRegistry {
    private static final Logger log = LogManager.getLogger(TrustManagerRegistry.class);
    private final TruststoreCrudService truststoreCrudService;
    private volatile Map<String, X509ExtendedTrustManager> trustManagerMap;

    public TrustManagerRegistry(TruststoreCrudService truststoreCrudService) {
        this.truststoreCrudService = Objects.requireNonNull(truststoreCrudService, "Truststore crud service is required");
        this.trustManagerMap = Collections.synchronizedMap(new HashMap());
        log.info("Truststore registry service created");
    }

    public void onTruststoreUpdate(String truststoreId, String operationType) {
        log.debug("Notification about operation '{}' on truststore '{}' received.", (Object)operationType, (Object)truststoreId);
        try {
            Optional<TruststoreData> truststoreData = this.truststoreCrudService.findOneById(truststoreId);
            if (truststoreData.isPresent()) {
                this.trustManagerMap.put(truststoreId, this.truststoreDataToTrustManager(truststoreData.get()));
            } else {
                this.trustManagerMap.remove(truststoreId);
                log.info("Truststore with id '{}' not found. Corresponding trust manager was removed.", (Object)truststoreId);
            }
            if (log.isInfoEnabled()) {
                String ids = this.getAvailableTrustManagersIds();
                log.info("Trust managers available after trust store updates: '{}'", (Object)ids);
            }
        }
        catch (ConfigValidationException | CannotCreateTrustManagerException | KeyStoreException | NoSuchAlgorithmException ex) {
            if (log.isDebugEnabled()) {
                String ids = this.getAvailableTrustManagersIds();
                log.debug("Cannot create trust manager for truststore '{}', available trust managers '{}'.", (Object)truststoreId, (Object)ids, (Object)ex);
            }
            throw new RuntimeException("Cannot update trust manager after operation '" + operationType + "' on trust store '" + truststoreId + "'.", ex);
        }
    }

    public void reloadAll() {
        List<TruststoreData> truststores = this.truststoreCrudService.loadAll();
        log.info("Loaded '{}' trust stores to init cache.", (Object)truststores.size());
        HashMap<String, X509ExtendedTrustManager> trustManagers = new HashMap<String, X509ExtendedTrustManager>();
        for (TruststoreData truststoreData : truststores) {
            try {
                X509ExtendedTrustManager x509TrustManager = this.truststoreDataToTrustManager(truststoreData);
                trustManagers.put(truststoreData.getId(), x509TrustManager);
            }
            catch (ConfigValidationException | CannotCreateTrustManagerException | KeyStoreException | NoSuchAlgorithmException e) {
                log.error("Cannot parse certificates in truststore '{}' or create trust manager. Truststore will be not available. Please check truststore data.", (Object)truststoreData.getId(), (Object)e);
            }
        }
        this.trustManagerMap = Collections.synchronizedMap(trustManagers);
        if (log.isInfoEnabled()) {
            String ids = this.getAvailableTrustManagersIds();
            log.info("Reloaded all trust stores and created trust managers, available trust managers: '{}'", (Object)ids);
        }
    }

    private X509ExtendedTrustManager truststoreDataToTrustManager(TruststoreData truststoreData) throws ConfigValidationException, KeyStoreException, NoSuchAlgorithmException, CannotCreateTrustManagerException {
        Collection<? extends Certificate> certificates = CertificatesParser.parseCertificates(truststoreData.getPem());
        KeyStore truststore = CertificatesParser.toTruststore(truststoreData.getId(), certificates);
        TrustManagerFactory trustManagerFactory = TrustManagerFactory.getInstance(TrustManagerFactory.getDefaultAlgorithm());
        trustManagerFactory.init(truststore);
        TrustManager[] trustManagers = trustManagerFactory.getTrustManagers();
        List x509TrustManagers = Arrays.stream(trustManagers).filter(X509ExtendedTrustManager.class::isInstance).collect(Collectors.toList());
        if (x509TrustManagers.size() != 1) {
            throw new CannotCreateTrustManagerException("Incorrect number of x509 trust managers: " + x509TrustManagers.size());
        }
        return (X509ExtendedTrustManager)x509TrustManagers.get(0);
    }

    public Optional<X509ExtendedTrustManager> findTrustManager(String truststoreId) {
        Objects.requireNonNull(truststoreId, "Truststore id must not be null");
        Optional<X509ExtendedTrustManager> x509TrustManager = Optional.ofNullable(this.trustManagerMap.get(truststoreId));
        log.trace("Trust manager loaded by id '{}' is '{}'.", (Object)truststoreId, x509TrustManager);
        return x509TrustManager;
    }

    private String getAvailableTrustManagersIds() {
        return new HashSet<String>(this.trustManagerMap.keySet()).stream().sorted().collect(Collectors.joining(", "));
    }
}

