"use strict";

var _interopRequireDefault = require("@babel/runtime/helpers/interopRequireDefault");
Object.defineProperty(exports, "__esModule", {
  value: true
});
exports.MultitenancyLifecycle = void 0;
var _defineProperty2 = _interopRequireDefault(require("@babel/runtime/helpers/defineProperty"));
var _lodash = require("lodash");
var _router = require("../../../../../src/core/server/http/router");
/*
 *    Copyright 2020 floragunn GmbH
 *
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 * http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 */

class MultitenancyLifecycle {
  constructor({
    authManager,
    kerberos,
    searchGuardBackend,
    configService,
    sessionStorageFactory,
    logger,
    pluginDependencies,
    spacesService,
    tenantService
  }) {
    (0, _defineProperty2.default)(this, "onPreAuth", async (request, response, toolkit) => {
      let authHeaders = request.headers;
      let sessionCookie;
      if (this.authManager) {
        const authInstance = await this.authManager.getAuthInstanceByRequest({
          request
        });
        if (authInstance) {
          sessionCookie = await authInstance.getCookieWithCredentials(request);
          authHeaders = authInstance.getAuthHeader(sessionCookie);
        } else {
          sessionCookie = await this.sessionStorageFactory.asScoped(request).get();
        }
      } else if (this.kerberos) {
        sessionCookie = await this.kerberos.getCookieWithCredentials(request);
        authHeaders = this.kerberos.getAuthHeader(sessionCookie);
      } else {
        sessionCookie = await this.sessionStorageFactory.asScoped(request).get();
      }
      const selectedTenant = await this.getSelectedTenant({
        request,
        authHeaders,
        sessionCookie
      });
      if (selectedTenant !== null) {
        const rawRequest = (0, _router.ensureRawRequest)(request);
        (0, _lodash.assign)(rawRequest.headers, authHeaders, {
          sgtenant: selectedTenant || ''
        });
        await this.tenantService.createIndexForTenant({
          request,
          selectedTenant
        });
        if (this.pluginDependencies.spaces) {
          // If we have a new tenant with no default space, we need to create it.
          await this.spacesService.createDefaultSpace({
            request,
            selectedTenant
          });
        }
      }
      return toolkit.next();
    });
    (0, _defineProperty2.default)(this, "isRelevantPath", request => {
      const path = request.url.pathname;

      // MT is only relevant for these paths
      const relevantPaths = ['/internal', '/goto', '/opensearch', '/app', '/api'];

      // MT is not relevant in these patterns
      const ignorePatterns = ['/api/status', '/api/v1/auth/config', '/api/v1/auth/login', '/api/v1/systeminfo'];
      return path === '/' || relevantPaths.some(root => path.startsWith(root)) && !ignorePatterns.some(root => path.startsWith(root));
    });
    /**
     * Get and validate the selected tenant.
     * @param request
     * @param authHeaders
     * @param sessionCookie
     * @returns {Promise<string|null>}
     */
    (0, _defineProperty2.default)(this, "getSelectedTenant", async ({
      request,
      authHeaders,
      sessionCookie
    }) => {
      const globalEnabled = this.configService.get('searchguard.multitenancy.tenants.enable_global');
      const privateEnabled = this.configService.get('searchguard.multitenancy.tenants.enable_private');
      const preferredTenants = this.configService.get('searchguard.multitenancy.tenants.preferred');
      const debugEnabled = this.configService.get('searchguard.multitenancy.debug');
      const backend = this.searchGuardBackend;

      // Make sure we have a sessionCookie object to work with
      if (!sessionCookie) {
        sessionCookie = {};
      }

      // default is the tenant stored in the tenants cookie
      let selectedTenant = sessionCookie && typeof sessionCookie.tenant !== 'undefined' ? sessionCookie.tenant : null;
      if (debugEnabled) {
        this.logger.info(`Multitenancy: tenant_storagecookie ${selectedTenant}`);
      }
      const externalTenant = this.getExternalTenant(request, debugEnabled);

      // MT is only relevant for some paths.
      // If we have an externalTenant, we need to update the cookie
      // though. Otherwise, if there's a redirect, the query parameter may
      // get lost before we can use it.
      if (!this.isRelevantPath(request) && !externalTenant) {
        return null;
      }
      if (externalTenant) {
        selectedTenant = externalTenant;
      }
      let authInfoResponse;
      try {
        // We need the user's data from the backend to validate the selected tenant
        authInfoResponse = await backend.authinfo(authHeaders);
      } catch (error) {
        this.logger.error(`Multitenancy: Could not get authinfo from ${request.url.pathname}. ${error}`);
        return null;
      }

      // if we have a tenant, check validity and set it
      if (typeof selectedTenant !== 'undefined' && selectedTenant !== null) {
        selectedTenant = backend.validateTenant(authInfoResponse.user_name, selectedTenant, authInfoResponse.sg_tenants, globalEnabled, privateEnabled);
      } else {
        // no tenant, choose configured, preferred tenant
        try {
          selectedTenant = backend.getTenantByPreference(request, authInfoResponse.user_name, authInfoResponse.sg_tenants, preferredTenants, globalEnabled, privateEnabled);
        } catch (error) {
          // nothing
        }
      }
      if (selectedTenant !== sessionCookie.tenant) {
        // save validated tenant in the cookie
        sessionCookie.tenant = selectedTenant;
        await this.sessionStorageFactory.asScoped(request).set(sessionCookie);
      }
      if (debugEnabled) {
        this.logger.info(`Multitenancy: tenant_assigned ${selectedTenant}`);
      }
      return selectedTenant;
    });
    /**
     * Check if we have a tenant set as query parameter
     * or as header value
     *
     * @param request
     * @param debugEnabled
     * @returns {null}
     */
    (0, _defineProperty2.default)(this, "getExternalTenant", (request, debugEnabled = false) => {
      let externalTenant = null;
      // check for tenant information in HTTP header. E.g. when using the saved objects API
      if (request.headers.sgtenant || request.headers.sg_tenant) {
        externalTenant = request.headers.sgtenant ? request.headers.sgtenant : request.headers.sg_tenant;
        if (debugEnabled) {
          this.logger.info(`Multitenancy: tenant_http_header' ${externalTenant}`);
        }
      }
      // check for tenant information in GET parameter. E.g. when using a share link. Overwrites the HTTP header.
      if (request.url.searchParams.has('sg_tenant') || request.url.searchParams.has('sgtenant')) {
        externalTenant = request.url.searchParams.has('sg_tenant') ? request.url.searchParams.get('sg_tenant') : request.url.searchParams.get('sgtenant');
        if (debugEnabled) {
          this.logger.info(`Multitenancy: tenant_url_param' ${externalTenant}`);
        }
      }
      return externalTenant;
    });
    this.authManager = authManager;
    this.searchGuardBackend = searchGuardBackend;
    this.configService = configService;
    this.sessionStorageFactory = sessionStorageFactory;
    this.logger = logger;
    this.pluginDependencies = pluginDependencies;
    this.spacesService = spacesService;
    this.tenantService = tenantService;
    this.kerberos = kerberos;
  }
}
exports.MultitenancyLifecycle = MultitenancyLifecycle;
//# sourceMappingURL=data:application/json;charset=utf-8;base64,eyJ2ZXJzaW9uIjozLCJuYW1lcyI6WyJfbG9kYXNoIiwicmVxdWlyZSIsIl9yb3V0ZXIiLCJNdWx0aXRlbmFuY3lMaWZlY3ljbGUiLCJjb25zdHJ1Y3RvciIsImF1dGhNYW5hZ2VyIiwia2VyYmVyb3MiLCJzZWFyY2hHdWFyZEJhY2tlbmQiLCJjb25maWdTZXJ2aWNlIiwic2Vzc2lvblN0b3JhZ2VGYWN0b3J5IiwibG9nZ2VyIiwicGx1Z2luRGVwZW5kZW5jaWVzIiwic3BhY2VzU2VydmljZSIsInRlbmFudFNlcnZpY2UiLCJfZGVmaW5lUHJvcGVydHkyIiwiZGVmYXVsdCIsInJlcXVlc3QiLCJyZXNwb25zZSIsInRvb2xraXQiLCJhdXRoSGVhZGVycyIsImhlYWRlcnMiLCJzZXNzaW9uQ29va2llIiwiYXV0aEluc3RhbmNlIiwiZ2V0QXV0aEluc3RhbmNlQnlSZXF1ZXN0IiwiZ2V0Q29va2llV2l0aENyZWRlbnRpYWxzIiwiZ2V0QXV0aEhlYWRlciIsImFzU2NvcGVkIiwiZ2V0Iiwic2VsZWN0ZWRUZW5hbnQiLCJnZXRTZWxlY3RlZFRlbmFudCIsInJhd1JlcXVlc3QiLCJlbnN1cmVSYXdSZXF1ZXN0IiwiYXNzaWduIiwic2d0ZW5hbnQiLCJjcmVhdGVJbmRleEZvclRlbmFudCIsInNwYWNlcyIsImNyZWF0ZURlZmF1bHRTcGFjZSIsIm5leHQiLCJwYXRoIiwidXJsIiwicGF0aG5hbWUiLCJyZWxldmFudFBhdGhzIiwiaWdub3JlUGF0dGVybnMiLCJzb21lIiwicm9vdCIsInN0YXJ0c1dpdGgiLCJnbG9iYWxFbmFibGVkIiwicHJpdmF0ZUVuYWJsZWQiLCJwcmVmZXJyZWRUZW5hbnRzIiwiZGVidWdFbmFibGVkIiwiYmFja2VuZCIsInRlbmFudCIsImluZm8iLCJleHRlcm5hbFRlbmFudCIsImdldEV4dGVybmFsVGVuYW50IiwiaXNSZWxldmFudFBhdGgiLCJhdXRoSW5mb1Jlc3BvbnNlIiwiYXV0aGluZm8iLCJlcnJvciIsInZhbGlkYXRlVGVuYW50IiwidXNlcl9uYW1lIiwic2dfdGVuYW50cyIsImdldFRlbmFudEJ5UHJlZmVyZW5jZSIsInNldCIsInNnX3RlbmFudCIsInNlYXJjaFBhcmFtcyIsImhhcyIsImV4cG9ydHMiXSwic291cmNlcyI6WyJtdWx0aXRlbmFuY3lfbGlmZWN5Y2xlLmpzIl0sInNvdXJjZXNDb250ZW50IjpbIi8qXG4gKiAgICBDb3B5cmlnaHQgMjAyMCBmbG9yYWd1bm4gR21iSFxuICpcbiAqIExpY2Vuc2VkIHVuZGVyIHRoZSBBcGFjaGUgTGljZW5zZSwgVmVyc2lvbiAyLjAgKHRoZSBcIkxpY2Vuc2VcIik7XG4gKiB5b3UgbWF5IG5vdCB1c2UgdGhpcyBmaWxlIGV4Y2VwdCBpbiBjb21wbGlhbmNlIHdpdGggdGhlIExpY2Vuc2UuXG4gKiBZb3UgbWF5IG9idGFpbiBhIGNvcHkgb2YgdGhlIExpY2Vuc2UgYXRcbiAqXG4gKiBodHRwOi8vd3d3LmFwYWNoZS5vcmcvbGljZW5zZXMvTElDRU5TRS0yLjBcbiAqXG4gKiBVbmxlc3MgcmVxdWlyZWQgYnkgYXBwbGljYWJsZSBsYXcgb3IgYWdyZWVkIHRvIGluIHdyaXRpbmcsIHNvZnR3YXJlXG4gKiBkaXN0cmlidXRlZCB1bmRlciB0aGUgTGljZW5zZSBpcyBkaXN0cmlidXRlZCBvbiBhbiBcIkFTIElTXCIgQkFTSVMsXG4gKiBXSVRIT1VUIFdBUlJBTlRJRVMgT1IgQ09ORElUSU9OUyBPRiBBTlkgS0lORCwgZWl0aGVyIGV4cHJlc3Mgb3IgaW1wbGllZC5cbiAqIFNlZSB0aGUgTGljZW5zZSBmb3IgdGhlIHNwZWNpZmljIGxhbmd1YWdlIGdvdmVybmluZyBwZXJtaXNzaW9ucyBhbmRcbiAqIGxpbWl0YXRpb25zIHVuZGVyIHRoZSBMaWNlbnNlLlxuICovXG5cbmltcG9ydCB7IGFzc2lnbiB9IGZyb20gJ2xvZGFzaCc7XG5pbXBvcnQgeyBlbnN1cmVSYXdSZXF1ZXN0IH0gZnJvbSAnLi4vLi4vLi4vLi4vLi4vc3JjL2NvcmUvc2VydmVyL2h0dHAvcm91dGVyJztcblxuZXhwb3J0IGNsYXNzIE11bHRpdGVuYW5jeUxpZmVjeWNsZSB7XG4gIGNvbnN0cnVjdG9yKHtcbiAgICBhdXRoTWFuYWdlcixcbiAgICBrZXJiZXJvcyxcbiAgICBzZWFyY2hHdWFyZEJhY2tlbmQsXG4gICAgY29uZmlnU2VydmljZSxcbiAgICBzZXNzaW9uU3RvcmFnZUZhY3RvcnksXG4gICAgbG9nZ2VyLFxuICAgIHBsdWdpbkRlcGVuZGVuY2llcyxcbiAgICBzcGFjZXNTZXJ2aWNlLFxuICAgIHRlbmFudFNlcnZpY2UsXG4gIH0pIHtcbiAgICB0aGlzLmF1dGhNYW5hZ2VyID0gYXV0aE1hbmFnZXI7XG4gICAgdGhpcy5zZWFyY2hHdWFyZEJhY2tlbmQgPSBzZWFyY2hHdWFyZEJhY2tlbmQ7XG4gICAgdGhpcy5jb25maWdTZXJ2aWNlID0gY29uZmlnU2VydmljZTtcbiAgICB0aGlzLnNlc3Npb25TdG9yYWdlRmFjdG9yeSA9IHNlc3Npb25TdG9yYWdlRmFjdG9yeTtcbiAgICB0aGlzLmxvZ2dlciA9IGxvZ2dlcjtcbiAgICB0aGlzLnBsdWdpbkRlcGVuZGVuY2llcyA9IHBsdWdpbkRlcGVuZGVuY2llcztcbiAgICB0aGlzLnNwYWNlc1NlcnZpY2UgPSBzcGFjZXNTZXJ2aWNlO1xuICAgIHRoaXMudGVuYW50U2VydmljZSA9IHRlbmFudFNlcnZpY2U7XG4gICAgdGhpcy5rZXJiZXJvcyA9IGtlcmJlcm9zO1xuICB9XG5cbiAgb25QcmVBdXRoID0gYXN5bmMgKHJlcXVlc3QsIHJlc3BvbnNlLCB0b29sa2l0KSA9PiB7XG4gICAgbGV0IGF1dGhIZWFkZXJzID0gcmVxdWVzdC5oZWFkZXJzO1xuXHRsZXQgc2Vzc2lvbkNvb2tpZTtcblxuXHRpZiAodGhpcy5hdXRoTWFuYWdlcikge1xuXHRcdCAgY29uc3QgYXV0aEluc3RhbmNlID0gYXdhaXQgdGhpcy5hdXRoTWFuYWdlci5nZXRBdXRoSW5zdGFuY2VCeVJlcXVlc3QoeyByZXF1ZXN0IH0pO1xuXG5cdFx0ICBpZiAoYXV0aEluc3RhbmNlKSB7XG5cdFx0XHQgIHNlc3Npb25Db29raWUgPSBhd2FpdCBhdXRoSW5zdGFuY2UuZ2V0Q29va2llV2l0aENyZWRlbnRpYWxzKHJlcXVlc3QpO1xuXHRcdFx0ICBhdXRoSGVhZGVycyA9IGF1dGhJbnN0YW5jZS5nZXRBdXRoSGVhZGVyKHNlc3Npb25Db29raWUpO1xuXHRcdCAgfSBlbHNlIHtcblx0XHRcdCAgc2Vzc2lvbkNvb2tpZSA9IGF3YWl0IHRoaXMuc2Vzc2lvblN0b3JhZ2VGYWN0b3J5LmFzU2NvcGVkKHJlcXVlc3QpLmdldCgpO1xuXHRcdCAgfVxuICAgIH0gZWxzZSBpZiAodGhpcy5rZXJiZXJvcykge1xuICAgICAgIHNlc3Npb25Db29raWUgPSBhd2FpdCB0aGlzLmtlcmJlcm9zLmdldENvb2tpZVdpdGhDcmVkZW50aWFscyhyZXF1ZXN0KTtcbiAgICAgICBhdXRoSGVhZGVycyA9IHRoaXMua2VyYmVyb3MuZ2V0QXV0aEhlYWRlcihzZXNzaW9uQ29va2llKTsgICAgICAgIFxuICAgIH0gZWxzZSB7XG4gICAgICAgc2Vzc2lvbkNvb2tpZSA9IGF3YWl0IHRoaXMuc2Vzc2lvblN0b3JhZ2VGYWN0b3J5LmFzU2NvcGVkKHJlcXVlc3QpLmdldCgpO1xuXHR9XG5cbiAgICBjb25zdCBzZWxlY3RlZFRlbmFudCA9IGF3YWl0IHRoaXMuZ2V0U2VsZWN0ZWRUZW5hbnQoe1xuICAgICAgcmVxdWVzdCxcbiAgICAgIGF1dGhIZWFkZXJzLFxuICAgICAgc2Vzc2lvbkNvb2tpZSxcbiAgICB9KTtcblxuICAgIGlmIChzZWxlY3RlZFRlbmFudCAhPT0gbnVsbCkge1xuICAgICAgY29uc3QgcmF3UmVxdWVzdCA9IGVuc3VyZVJhd1JlcXVlc3QocmVxdWVzdCk7XG4gICAgICBhc3NpZ24ocmF3UmVxdWVzdC5oZWFkZXJzLCBhdXRoSGVhZGVycywgeyBzZ3RlbmFudDogc2VsZWN0ZWRUZW5hbnQgfHwgJycgfSk7XG5cbiAgICAgIGF3YWl0IHRoaXMudGVuYW50U2VydmljZS5jcmVhdGVJbmRleEZvclRlbmFudCh7IHJlcXVlc3QsIHNlbGVjdGVkVGVuYW50IH0pO1xuXG4gICAgICBpZiAodGhpcy5wbHVnaW5EZXBlbmRlbmNpZXMuc3BhY2VzKSB7XG4gICAgICAgIC8vIElmIHdlIGhhdmUgYSBuZXcgdGVuYW50IHdpdGggbm8gZGVmYXVsdCBzcGFjZSwgd2UgbmVlZCB0byBjcmVhdGUgaXQuXG4gICAgICAgIGF3YWl0IHRoaXMuc3BhY2VzU2VydmljZS5jcmVhdGVEZWZhdWx0U3BhY2UoeyByZXF1ZXN0LCBzZWxlY3RlZFRlbmFudCB9KTtcbiAgICAgIH1cbiAgICB9XG5cbiAgICByZXR1cm4gdG9vbGtpdC5uZXh0KCk7XG4gIH07XG5cbiAgaXNSZWxldmFudFBhdGggPSAocmVxdWVzdCkgPT4ge1xuICAgIGNvbnN0IHBhdGggPSByZXF1ZXN0LnVybC5wYXRobmFtZTtcblxuICAgIC8vIE1UIGlzIG9ubHkgcmVsZXZhbnQgZm9yIHRoZXNlIHBhdGhzXG4gICAgY29uc3QgcmVsZXZhbnRQYXRocyA9IFtcbiAgICAgICcvaW50ZXJuYWwnLFxuICAgICAgJy9nb3RvJyxcbiAgICAgICcvb3BlbnNlYXJjaCcsXG4gICAgICAnL2FwcCcsXG4gICAgICAnL2FwaSdcbiAgICBdO1xuXG4gICAgLy8gTVQgaXMgbm90IHJlbGV2YW50IGluIHRoZXNlIHBhdHRlcm5zXG4gICAgY29uc3QgaWdub3JlUGF0dGVybnMgPSBbXG4gICAgICAnL2FwaS9zdGF0dXMnLFxuICAgICAgJy9hcGkvdjEvYXV0aC9jb25maWcnLFxuICAgICAgJy9hcGkvdjEvYXV0aC9sb2dpbicsXG4gICAgICAnL2FwaS92MS9zeXN0ZW1pbmZvJyxcbiAgICBdXG5cbiAgICByZXR1cm4gcGF0aCA9PT0gJy8nIHx8IChcbiAgICAgIHJlbGV2YW50UGF0aHMuc29tZShyb290ID0+IHBhdGguc3RhcnRzV2l0aChyb290KSkgJiZcbiAgICAgICFpZ25vcmVQYXR0ZXJucy5zb21lKHJvb3QgPT4gcGF0aC5zdGFydHNXaXRoKHJvb3QpKVxuICAgICk7XG4gIH07XG5cbiAgLyoqXG4gICAqIEdldCBhbmQgdmFsaWRhdGUgdGhlIHNlbGVjdGVkIHRlbmFudC5cbiAgICogQHBhcmFtIHJlcXVlc3RcbiAgICogQHBhcmFtIGF1dGhIZWFkZXJzXG4gICAqIEBwYXJhbSBzZXNzaW9uQ29va2llXG4gICAqIEByZXR1cm5zIHtQcm9taXNlPHN0cmluZ3xudWxsPn1cbiAgICovXG4gIGdldFNlbGVjdGVkVGVuYW50ID0gYXN5bmMgKHsgcmVxdWVzdCwgYXV0aEhlYWRlcnMsIHNlc3Npb25Db29raWUgfSkgPT4ge1xuICAgIGNvbnN0IGdsb2JhbEVuYWJsZWQgPSB0aGlzLmNvbmZpZ1NlcnZpY2UuZ2V0KCdzZWFyY2hndWFyZC5tdWx0aXRlbmFuY3kudGVuYW50cy5lbmFibGVfZ2xvYmFsJyk7XG4gICAgY29uc3QgcHJpdmF0ZUVuYWJsZWQgPSB0aGlzLmNvbmZpZ1NlcnZpY2UuZ2V0KFxuICAgICAgJ3NlYXJjaGd1YXJkLm11bHRpdGVuYW5jeS50ZW5hbnRzLmVuYWJsZV9wcml2YXRlJ1xuICAgICk7XG4gICAgY29uc3QgcHJlZmVycmVkVGVuYW50cyA9IHRoaXMuY29uZmlnU2VydmljZS5nZXQoJ3NlYXJjaGd1YXJkLm11bHRpdGVuYW5jeS50ZW5hbnRzLnByZWZlcnJlZCcpO1xuICAgIGNvbnN0IGRlYnVnRW5hYmxlZCA9IHRoaXMuY29uZmlnU2VydmljZS5nZXQoJ3NlYXJjaGd1YXJkLm11bHRpdGVuYW5jeS5kZWJ1ZycpO1xuICAgIGNvbnN0IGJhY2tlbmQgPSB0aGlzLnNlYXJjaEd1YXJkQmFja2VuZDtcblxuICAgIC8vIE1ha2Ugc3VyZSB3ZSBoYXZlIGEgc2Vzc2lvbkNvb2tpZSBvYmplY3QgdG8gd29yayB3aXRoXG4gICAgaWYgKCFzZXNzaW9uQ29va2llKSB7XG4gICAgICBzZXNzaW9uQ29va2llID0ge307XG4gICAgfVxuXG4gICAgLy8gZGVmYXVsdCBpcyB0aGUgdGVuYW50IHN0b3JlZCBpbiB0aGUgdGVuYW50cyBjb29raWVcbiAgICBsZXQgc2VsZWN0ZWRUZW5hbnQgPVxuICAgICAgc2Vzc2lvbkNvb2tpZSAmJiB0eXBlb2Ygc2Vzc2lvbkNvb2tpZS50ZW5hbnQgIT09ICd1bmRlZmluZWQnID8gc2Vzc2lvbkNvb2tpZS50ZW5hbnQgOiBudWxsO1xuXG4gICAgaWYgKGRlYnVnRW5hYmxlZCkge1xuICAgICAgdGhpcy5sb2dnZXIuaW5mbyhgTXVsdGl0ZW5hbmN5OiB0ZW5hbnRfc3RvcmFnZWNvb2tpZSAke3NlbGVjdGVkVGVuYW50fWApO1xuICAgIH1cblxuICAgIGNvbnN0IGV4dGVybmFsVGVuYW50ID0gdGhpcy5nZXRFeHRlcm5hbFRlbmFudChyZXF1ZXN0LCBkZWJ1Z0VuYWJsZWQpO1xuXG4gICAgLy8gTVQgaXMgb25seSByZWxldmFudCBmb3Igc29tZSBwYXRocy5cbiAgICAvLyBJZiB3ZSBoYXZlIGFuIGV4dGVybmFsVGVuYW50LCB3ZSBuZWVkIHRvIHVwZGF0ZSB0aGUgY29va2llXG4gICAgLy8gdGhvdWdoLiBPdGhlcndpc2UsIGlmIHRoZXJlJ3MgYSByZWRpcmVjdCwgdGhlIHF1ZXJ5IHBhcmFtZXRlciBtYXlcbiAgICAvLyBnZXQgbG9zdCBiZWZvcmUgd2UgY2FuIHVzZSBpdC5cbiAgICBpZiAoIXRoaXMuaXNSZWxldmFudFBhdGgocmVxdWVzdCkgJiYgIWV4dGVybmFsVGVuYW50KSB7XG4gICAgICByZXR1cm4gbnVsbDtcbiAgICB9XG5cbiAgICBpZiAoZXh0ZXJuYWxUZW5hbnQpIHtcbiAgICAgIHNlbGVjdGVkVGVuYW50ID0gZXh0ZXJuYWxUZW5hbnQ7XG4gICAgfVxuXG4gICAgbGV0IGF1dGhJbmZvUmVzcG9uc2U7XG4gICAgdHJ5IHtcbiAgICAgIC8vIFdlIG5lZWQgdGhlIHVzZXIncyBkYXRhIGZyb20gdGhlIGJhY2tlbmQgdG8gdmFsaWRhdGUgdGhlIHNlbGVjdGVkIHRlbmFudFxuICAgICAgYXV0aEluZm9SZXNwb25zZSA9IGF3YWl0IGJhY2tlbmQuYXV0aGluZm8oYXV0aEhlYWRlcnMpO1xuICAgIH0gY2F0Y2ggKGVycm9yKSB7XG4gICAgICB0aGlzLmxvZ2dlci5lcnJvcihgTXVsdGl0ZW5hbmN5OiBDb3VsZCBub3QgZ2V0IGF1dGhpbmZvIGZyb20gJHtyZXF1ZXN0LnVybC5wYXRobmFtZX0uICR7ZXJyb3J9YCk7XG4gICAgICByZXR1cm4gbnVsbDtcbiAgICB9XG5cbiAgICAvLyBpZiB3ZSBoYXZlIGEgdGVuYW50LCBjaGVjayB2YWxpZGl0eSBhbmQgc2V0IGl0XG4gICAgaWYgKHR5cGVvZiBzZWxlY3RlZFRlbmFudCAhPT0gJ3VuZGVmaW5lZCcgJiYgc2VsZWN0ZWRUZW5hbnQgIT09IG51bGwpIHtcbiAgICAgIHNlbGVjdGVkVGVuYW50ID0gYmFja2VuZC52YWxpZGF0ZVRlbmFudChcbiAgICAgICAgYXV0aEluZm9SZXNwb25zZS51c2VyX25hbWUsXG4gICAgICAgIHNlbGVjdGVkVGVuYW50LFxuICAgICAgICBhdXRoSW5mb1Jlc3BvbnNlLnNnX3RlbmFudHMsXG4gICAgICAgIGdsb2JhbEVuYWJsZWQsXG4gICAgICAgIHByaXZhdGVFbmFibGVkXG4gICAgICApO1xuICAgIH0gZWxzZSB7XG4gICAgICAvLyBubyB0ZW5hbnQsIGNob29zZSBjb25maWd1cmVkLCBwcmVmZXJyZWQgdGVuYW50XG4gICAgICB0cnkge1xuICAgICAgICBzZWxlY3RlZFRlbmFudCA9IGJhY2tlbmQuZ2V0VGVuYW50QnlQcmVmZXJlbmNlKFxuICAgICAgICAgIHJlcXVlc3QsXG4gICAgICAgICAgYXV0aEluZm9SZXNwb25zZS51c2VyX25hbWUsXG4gICAgICAgICAgYXV0aEluZm9SZXNwb25zZS5zZ190ZW5hbnRzLFxuICAgICAgICAgIHByZWZlcnJlZFRlbmFudHMsXG4gICAgICAgICAgZ2xvYmFsRW5hYmxlZCxcbiAgICAgICAgICBwcml2YXRlRW5hYmxlZFxuICAgICAgICApO1xuICAgICAgfSBjYXRjaCAoZXJyb3IpIHtcbiAgICAgICAgLy8gbm90aGluZ1xuICAgICAgfVxuICAgIH1cblxuICAgIGlmIChzZWxlY3RlZFRlbmFudCAhPT0gc2Vzc2lvbkNvb2tpZS50ZW5hbnQpIHtcbiAgICAgIC8vIHNhdmUgdmFsaWRhdGVkIHRlbmFudCBpbiB0aGUgY29va2llXG4gICAgICBzZXNzaW9uQ29va2llLnRlbmFudCA9IHNlbGVjdGVkVGVuYW50O1xuICAgICAgYXdhaXQgdGhpcy5zZXNzaW9uU3RvcmFnZUZhY3RvcnkuYXNTY29wZWQocmVxdWVzdCkuc2V0KHNlc3Npb25Db29raWUpO1xuICAgIH1cblxuICAgIGlmIChkZWJ1Z0VuYWJsZWQpIHtcbiAgICAgIHRoaXMubG9nZ2VyLmluZm8oYE11bHRpdGVuYW5jeTogdGVuYW50X2Fzc2lnbmVkICR7c2VsZWN0ZWRUZW5hbnR9YCk7XG4gICAgfVxuXG4gICAgcmV0dXJuIHNlbGVjdGVkVGVuYW50O1xuICB9O1xuXG4gIC8qKlxuICAgKiBDaGVjayBpZiB3ZSBoYXZlIGEgdGVuYW50IHNldCBhcyBxdWVyeSBwYXJhbWV0ZXJcbiAgICogb3IgYXMgaGVhZGVyIHZhbHVlXG4gICAqXG4gICAqIEBwYXJhbSByZXF1ZXN0XG4gICAqIEBwYXJhbSBkZWJ1Z0VuYWJsZWRcbiAgICogQHJldHVybnMge251bGx9XG4gICAqL1xuICBnZXRFeHRlcm5hbFRlbmFudCA9IChyZXF1ZXN0LCBkZWJ1Z0VuYWJsZWQgPSBmYWxzZSkgPT4ge1xuICAgIGxldCBleHRlcm5hbFRlbmFudCA9IG51bGw7XG4gICAgLy8gY2hlY2sgZm9yIHRlbmFudCBpbmZvcm1hdGlvbiBpbiBIVFRQIGhlYWRlci4gRS5nLiB3aGVuIHVzaW5nIHRoZSBzYXZlZCBvYmplY3RzIEFQSVxuICAgIGlmIChyZXF1ZXN0LmhlYWRlcnMuc2d0ZW5hbnQgfHwgcmVxdWVzdC5oZWFkZXJzLnNnX3RlbmFudCkge1xuICAgICAgZXh0ZXJuYWxUZW5hbnQgPSByZXF1ZXN0LmhlYWRlcnMuc2d0ZW5hbnRcbiAgICAgICAgPyByZXF1ZXN0LmhlYWRlcnMuc2d0ZW5hbnRcbiAgICAgICAgOiByZXF1ZXN0LmhlYWRlcnMuc2dfdGVuYW50O1xuXG4gICAgICBpZiAoZGVidWdFbmFibGVkKSB7XG4gICAgICAgIHRoaXMubG9nZ2VyLmluZm8oYE11bHRpdGVuYW5jeTogdGVuYW50X2h0dHBfaGVhZGVyJyAke2V4dGVybmFsVGVuYW50fWApO1xuICAgICAgfVxuICAgIH1cbiAgICAvLyBjaGVjayBmb3IgdGVuYW50IGluZm9ybWF0aW9uIGluIEdFVCBwYXJhbWV0ZXIuIEUuZy4gd2hlbiB1c2luZyBhIHNoYXJlIGxpbmsuIE92ZXJ3cml0ZXMgdGhlIEhUVFAgaGVhZGVyLlxuICAgIGlmIChyZXF1ZXN0LnVybC5zZWFyY2hQYXJhbXMuaGFzKCdzZ190ZW5hbnQnKSB8fCByZXF1ZXN0LnVybC5zZWFyY2hQYXJhbXMuaGFzKCdzZ3RlbmFudCcpKSB7XG4gICAgICBleHRlcm5hbFRlbmFudCA9IHJlcXVlc3QudXJsLnNlYXJjaFBhcmFtcy5oYXMoJ3NnX3RlbmFudCcpXG4gICAgICAgID8gcmVxdWVzdC51cmwuc2VhcmNoUGFyYW1zLmdldCgnc2dfdGVuYW50JylcbiAgICAgICAgOiByZXF1ZXN0LnVybC5zZWFyY2hQYXJhbXMuZ2V0KCdzZ3RlbmFudCcpO1xuXG4gICAgICBpZiAoZGVidWdFbmFibGVkKSB7XG4gICAgICAgIHRoaXMubG9nZ2VyLmluZm8oYE11bHRpdGVuYW5jeTogdGVuYW50X3VybF9wYXJhbScgJHtleHRlcm5hbFRlbmFudH1gKTtcbiAgICAgIH1cbiAgICB9XG5cbiAgICByZXR1cm4gZXh0ZXJuYWxUZW5hbnQ7XG4gIH07XG59XG4iXSwibWFwcGluZ3MiOiI7Ozs7Ozs7O0FBZ0JBLElBQUFBLE9BQUEsR0FBQUMsT0FBQTtBQUNBLElBQUFDLE9BQUEsR0FBQUQsT0FBQTtBQWpCQTtBQUNBO0FBQ0E7QUFDQTtBQUNBO0FBQ0E7QUFDQTtBQUNBO0FBQ0E7QUFDQTtBQUNBO0FBQ0E7QUFDQTtBQUNBO0FBQ0E7O0FBS08sTUFBTUUscUJBQXFCLENBQUM7RUFDakNDLFdBQVdBLENBQUM7SUFDVkMsV0FBVztJQUNYQyxRQUFRO0lBQ1JDLGtCQUFrQjtJQUNsQkMsYUFBYTtJQUNiQyxxQkFBcUI7SUFDckJDLE1BQU07SUFDTkMsa0JBQWtCO0lBQ2xCQyxhQUFhO0lBQ2JDO0VBQ0YsQ0FBQyxFQUFFO0lBQUEsSUFBQUMsZ0JBQUEsQ0FBQUMsT0FBQSxxQkFZUyxPQUFPQyxPQUFPLEVBQUVDLFFBQVEsRUFBRUMsT0FBTyxLQUFLO01BQ2hELElBQUlDLFdBQVcsR0FBR0gsT0FBTyxDQUFDSSxPQUFPO01BQ3BDLElBQUlDLGFBQWE7TUFFakIsSUFBSSxJQUFJLENBQUNoQixXQUFXLEVBQUU7UUFDbkIsTUFBTWlCLFlBQVksR0FBRyxNQUFNLElBQUksQ0FBQ2pCLFdBQVcsQ0FBQ2tCLHdCQUF3QixDQUFDO1VBQUVQO1FBQVEsQ0FBQyxDQUFDO1FBRWpGLElBQUlNLFlBQVksRUFBRTtVQUNqQkQsYUFBYSxHQUFHLE1BQU1DLFlBQVksQ0FBQ0Usd0JBQXdCLENBQUNSLE9BQU8sQ0FBQztVQUNwRUcsV0FBVyxHQUFHRyxZQUFZLENBQUNHLGFBQWEsQ0FBQ0osYUFBYSxDQUFDO1FBQ3hELENBQUMsTUFBTTtVQUNOQSxhQUFhLEdBQUcsTUFBTSxJQUFJLENBQUNaLHFCQUFxQixDQUFDaUIsUUFBUSxDQUFDVixPQUFPLENBQUMsQ0FBQ1csR0FBRyxDQUFDLENBQUM7UUFDekU7TUFDQSxDQUFDLE1BQU0sSUFBSSxJQUFJLENBQUNyQixRQUFRLEVBQUU7UUFDdkJlLGFBQWEsR0FBRyxNQUFNLElBQUksQ0FBQ2YsUUFBUSxDQUFDa0Isd0JBQXdCLENBQUNSLE9BQU8sQ0FBQztRQUNyRUcsV0FBVyxHQUFHLElBQUksQ0FBQ2IsUUFBUSxDQUFDbUIsYUFBYSxDQUFDSixhQUFhLENBQUM7TUFDM0QsQ0FBQyxNQUFNO1FBQ0pBLGFBQWEsR0FBRyxNQUFNLElBQUksQ0FBQ1oscUJBQXFCLENBQUNpQixRQUFRLENBQUNWLE9BQU8sQ0FBQyxDQUFDVyxHQUFHLENBQUMsQ0FBQztNQUM5RTtNQUVHLE1BQU1DLGNBQWMsR0FBRyxNQUFNLElBQUksQ0FBQ0MsaUJBQWlCLENBQUM7UUFDbERiLE9BQU87UUFDUEcsV0FBVztRQUNYRTtNQUNGLENBQUMsQ0FBQztNQUVGLElBQUlPLGNBQWMsS0FBSyxJQUFJLEVBQUU7UUFDM0IsTUFBTUUsVUFBVSxHQUFHLElBQUFDLHdCQUFnQixFQUFDZixPQUFPLENBQUM7UUFDNUMsSUFBQWdCLGNBQU0sRUFBQ0YsVUFBVSxDQUFDVixPQUFPLEVBQUVELFdBQVcsRUFBRTtVQUFFYyxRQUFRLEVBQUVMLGNBQWMsSUFBSTtRQUFHLENBQUMsQ0FBQztRQUUzRSxNQUFNLElBQUksQ0FBQ2YsYUFBYSxDQUFDcUIsb0JBQW9CLENBQUM7VUFBRWxCLE9BQU87VUFBRVk7UUFBZSxDQUFDLENBQUM7UUFFMUUsSUFBSSxJQUFJLENBQUNqQixrQkFBa0IsQ0FBQ3dCLE1BQU0sRUFBRTtVQUNsQztVQUNBLE1BQU0sSUFBSSxDQUFDdkIsYUFBYSxDQUFDd0Isa0JBQWtCLENBQUM7WUFBRXBCLE9BQU87WUFBRVk7VUFBZSxDQUFDLENBQUM7UUFDMUU7TUFDRjtNQUVBLE9BQU9WLE9BQU8sQ0FBQ21CLElBQUksQ0FBQyxDQUFDO0lBQ3ZCLENBQUM7SUFBQSxJQUFBdkIsZ0JBQUEsQ0FBQUMsT0FBQSwwQkFFaUJDLE9BQU8sSUFBSztNQUM1QixNQUFNc0IsSUFBSSxHQUFHdEIsT0FBTyxDQUFDdUIsR0FBRyxDQUFDQyxRQUFROztNQUVqQztNQUNBLE1BQU1DLGFBQWEsR0FBRyxDQUNwQixXQUFXLEVBQ1gsT0FBTyxFQUNQLGFBQWEsRUFDYixNQUFNLEVBQ04sTUFBTSxDQUNQOztNQUVEO01BQ0EsTUFBTUMsY0FBYyxHQUFHLENBQ3JCLGFBQWEsRUFDYixxQkFBcUIsRUFDckIsb0JBQW9CLEVBQ3BCLG9CQUFvQixDQUNyQjtNQUVELE9BQU9KLElBQUksS0FBSyxHQUFHLElBQ2pCRyxhQUFhLENBQUNFLElBQUksQ0FBQ0MsSUFBSSxJQUFJTixJQUFJLENBQUNPLFVBQVUsQ0FBQ0QsSUFBSSxDQUFDLENBQUMsSUFDakQsQ0FBQ0YsY0FBYyxDQUFDQyxJQUFJLENBQUNDLElBQUksSUFBSU4sSUFBSSxDQUFDTyxVQUFVLENBQUNELElBQUksQ0FBQyxDQUNuRDtJQUNILENBQUM7SUFFRDtBQUNGO0FBQ0E7QUFDQTtBQUNBO0FBQ0E7QUFDQTtJQU5FLElBQUE5QixnQkFBQSxDQUFBQyxPQUFBLDZCQU9vQixPQUFPO01BQUVDLE9BQU87TUFBRUcsV0FBVztNQUFFRTtJQUFjLENBQUMsS0FBSztNQUNyRSxNQUFNeUIsYUFBYSxHQUFHLElBQUksQ0FBQ3RDLGFBQWEsQ0FBQ21CLEdBQUcsQ0FBQyxnREFBZ0QsQ0FBQztNQUM5RixNQUFNb0IsY0FBYyxHQUFHLElBQUksQ0FBQ3ZDLGFBQWEsQ0FBQ21CLEdBQUcsQ0FDM0MsaURBQ0YsQ0FBQztNQUNELE1BQU1xQixnQkFBZ0IsR0FBRyxJQUFJLENBQUN4QyxhQUFhLENBQUNtQixHQUFHLENBQUMsNENBQTRDLENBQUM7TUFDN0YsTUFBTXNCLFlBQVksR0FBRyxJQUFJLENBQUN6QyxhQUFhLENBQUNtQixHQUFHLENBQUMsZ0NBQWdDLENBQUM7TUFDN0UsTUFBTXVCLE9BQU8sR0FBRyxJQUFJLENBQUMzQyxrQkFBa0I7O01BRXZDO01BQ0EsSUFBSSxDQUFDYyxhQUFhLEVBQUU7UUFDbEJBLGFBQWEsR0FBRyxDQUFDLENBQUM7TUFDcEI7O01BRUE7TUFDQSxJQUFJTyxjQUFjLEdBQ2hCUCxhQUFhLElBQUksT0FBT0EsYUFBYSxDQUFDOEIsTUFBTSxLQUFLLFdBQVcsR0FBRzlCLGFBQWEsQ0FBQzhCLE1BQU0sR0FBRyxJQUFJO01BRTVGLElBQUlGLFlBQVksRUFBRTtRQUNoQixJQUFJLENBQUN2QyxNQUFNLENBQUMwQyxJQUFJLENBQUMsc0NBQXNDeEIsY0FBYyxFQUFFLENBQUM7TUFDMUU7TUFFQSxNQUFNeUIsY0FBYyxHQUFHLElBQUksQ0FBQ0MsaUJBQWlCLENBQUN0QyxPQUFPLEVBQUVpQyxZQUFZLENBQUM7O01BRXBFO01BQ0E7TUFDQTtNQUNBO01BQ0EsSUFBSSxDQUFDLElBQUksQ0FBQ00sY0FBYyxDQUFDdkMsT0FBTyxDQUFDLElBQUksQ0FBQ3FDLGNBQWMsRUFBRTtRQUNwRCxPQUFPLElBQUk7TUFDYjtNQUVBLElBQUlBLGNBQWMsRUFBRTtRQUNsQnpCLGNBQWMsR0FBR3lCLGNBQWM7TUFDakM7TUFFQSxJQUFJRyxnQkFBZ0I7TUFDcEIsSUFBSTtRQUNGO1FBQ0FBLGdCQUFnQixHQUFHLE1BQU1OLE9BQU8sQ0FBQ08sUUFBUSxDQUFDdEMsV0FBVyxDQUFDO01BQ3hELENBQUMsQ0FBQyxPQUFPdUMsS0FBSyxFQUFFO1FBQ2QsSUFBSSxDQUFDaEQsTUFBTSxDQUFDZ0QsS0FBSyxDQUFDLDZDQUE2QzFDLE9BQU8sQ0FBQ3VCLEdBQUcsQ0FBQ0MsUUFBUSxLQUFLa0IsS0FBSyxFQUFFLENBQUM7UUFDaEcsT0FBTyxJQUFJO01BQ2I7O01BRUE7TUFDQSxJQUFJLE9BQU85QixjQUFjLEtBQUssV0FBVyxJQUFJQSxjQUFjLEtBQUssSUFBSSxFQUFFO1FBQ3BFQSxjQUFjLEdBQUdzQixPQUFPLENBQUNTLGNBQWMsQ0FDckNILGdCQUFnQixDQUFDSSxTQUFTLEVBQzFCaEMsY0FBYyxFQUNkNEIsZ0JBQWdCLENBQUNLLFVBQVUsRUFDM0JmLGFBQWEsRUFDYkMsY0FDRixDQUFDO01BQ0gsQ0FBQyxNQUFNO1FBQ0w7UUFDQSxJQUFJO1VBQ0ZuQixjQUFjLEdBQUdzQixPQUFPLENBQUNZLHFCQUFxQixDQUM1QzlDLE9BQU8sRUFDUHdDLGdCQUFnQixDQUFDSSxTQUFTLEVBQzFCSixnQkFBZ0IsQ0FBQ0ssVUFBVSxFQUMzQmIsZ0JBQWdCLEVBQ2hCRixhQUFhLEVBQ2JDLGNBQ0YsQ0FBQztRQUNILENBQUMsQ0FBQyxPQUFPVyxLQUFLLEVBQUU7VUFDZDtRQUFBO01BRUo7TUFFQSxJQUFJOUIsY0FBYyxLQUFLUCxhQUFhLENBQUM4QixNQUFNLEVBQUU7UUFDM0M7UUFDQTlCLGFBQWEsQ0FBQzhCLE1BQU0sR0FBR3ZCLGNBQWM7UUFDckMsTUFBTSxJQUFJLENBQUNuQixxQkFBcUIsQ0FBQ2lCLFFBQVEsQ0FBQ1YsT0FBTyxDQUFDLENBQUMrQyxHQUFHLENBQUMxQyxhQUFhLENBQUM7TUFDdkU7TUFFQSxJQUFJNEIsWUFBWSxFQUFFO1FBQ2hCLElBQUksQ0FBQ3ZDLE1BQU0sQ0FBQzBDLElBQUksQ0FBQyxpQ0FBaUN4QixjQUFjLEVBQUUsQ0FBQztNQUNyRTtNQUVBLE9BQU9BLGNBQWM7SUFDdkIsQ0FBQztJQUVEO0FBQ0Y7QUFDQTtBQUNBO0FBQ0E7QUFDQTtBQUNBO0FBQ0E7SUFQRSxJQUFBZCxnQkFBQSxDQUFBQyxPQUFBLDZCQVFvQixDQUFDQyxPQUFPLEVBQUVpQyxZQUFZLEdBQUcsS0FBSyxLQUFLO01BQ3JELElBQUlJLGNBQWMsR0FBRyxJQUFJO01BQ3pCO01BQ0EsSUFBSXJDLE9BQU8sQ0FBQ0ksT0FBTyxDQUFDYSxRQUFRLElBQUlqQixPQUFPLENBQUNJLE9BQU8sQ0FBQzRDLFNBQVMsRUFBRTtRQUN6RFgsY0FBYyxHQUFHckMsT0FBTyxDQUFDSSxPQUFPLENBQUNhLFFBQVEsR0FDckNqQixPQUFPLENBQUNJLE9BQU8sQ0FBQ2EsUUFBUSxHQUN4QmpCLE9BQU8sQ0FBQ0ksT0FBTyxDQUFDNEMsU0FBUztRQUU3QixJQUFJZixZQUFZLEVBQUU7VUFDaEIsSUFBSSxDQUFDdkMsTUFBTSxDQUFDMEMsSUFBSSxDQUFDLHFDQUFxQ0MsY0FBYyxFQUFFLENBQUM7UUFDekU7TUFDRjtNQUNBO01BQ0EsSUFBSXJDLE9BQU8sQ0FBQ3VCLEdBQUcsQ0FBQzBCLFlBQVksQ0FBQ0MsR0FBRyxDQUFDLFdBQVcsQ0FBQyxJQUFJbEQsT0FBTyxDQUFDdUIsR0FBRyxDQUFDMEIsWUFBWSxDQUFDQyxHQUFHLENBQUMsVUFBVSxDQUFDLEVBQUU7UUFDekZiLGNBQWMsR0FBR3JDLE9BQU8sQ0FBQ3VCLEdBQUcsQ0FBQzBCLFlBQVksQ0FBQ0MsR0FBRyxDQUFDLFdBQVcsQ0FBQyxHQUN0RGxELE9BQU8sQ0FBQ3VCLEdBQUcsQ0FBQzBCLFlBQVksQ0FBQ3RDLEdBQUcsQ0FBQyxXQUFXLENBQUMsR0FDekNYLE9BQU8sQ0FBQ3VCLEdBQUcsQ0FBQzBCLFlBQVksQ0FBQ3RDLEdBQUcsQ0FBQyxVQUFVLENBQUM7UUFFNUMsSUFBSXNCLFlBQVksRUFBRTtVQUNoQixJQUFJLENBQUN2QyxNQUFNLENBQUMwQyxJQUFJLENBQUMsbUNBQW1DQyxjQUFjLEVBQUUsQ0FBQztRQUN2RTtNQUNGO01BRUEsT0FBT0EsY0FBYztJQUN2QixDQUFDO0lBeE1DLElBQUksQ0FBQ2hELFdBQVcsR0FBR0EsV0FBVztJQUM5QixJQUFJLENBQUNFLGtCQUFrQixHQUFHQSxrQkFBa0I7SUFDNUMsSUFBSSxDQUFDQyxhQUFhLEdBQUdBLGFBQWE7SUFDbEMsSUFBSSxDQUFDQyxxQkFBcUIsR0FBR0EscUJBQXFCO0lBQ2xELElBQUksQ0FBQ0MsTUFBTSxHQUFHQSxNQUFNO0lBQ3BCLElBQUksQ0FBQ0Msa0JBQWtCLEdBQUdBLGtCQUFrQjtJQUM1QyxJQUFJLENBQUNDLGFBQWEsR0FBR0EsYUFBYTtJQUNsQyxJQUFJLENBQUNDLGFBQWEsR0FBR0EsYUFBYTtJQUNsQyxJQUFJLENBQUNQLFFBQVEsR0FBR0EsUUFBUTtFQUMxQjtBQWdNRjtBQUFDNkQsT0FBQSxDQUFBaEUscUJBQUEsR0FBQUEscUJBQUEiLCJpZ25vcmVMaXN0IjpbXX0=