"use strict";

var _interopRequireDefault = require("@babel/runtime/helpers/interopRequireDefault");
Object.defineProperty(exports, "__esModule", {
  value: true
});
exports.TenantService = void 0;
var _defineProperty2 = _interopRequireDefault(require("@babel/runtime/helpers/defineProperty"));
var _lodash = require("lodash");
var _tenants_migration_service = require("./tenants_migration_service");
var _migrations = require("../../../../../src/core/server/saved_objects/migrations");
/*
 *    Copyright 2020 floragunn GmbH
 *
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 * http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 */

class TenantService {
  constructor({
    clusterClient,
    logger,
    configService,
    savedObjects,
    coreContext
  }) {
    (0, _defineProperty2.default)(this, "logErrorDetails", (error, message) => {
      if (!message) message = error.message;
      let errorMeta = JSON.stringify((0, _lodash.get)(error, 'meta.body', ''));
      if (!errorMeta || !errorMeta.length) errorMeta = error;
      this.logger.error(`${message}, ${errorMeta}`);
    });
    (0, _defineProperty2.default)(this, "indexExists", async ({
      request,
      indexName
    }) => {
      try {
        await this.clusterClient.asScoped(request).asCurrentUser.transport.request({
          method: 'get',
          path: `/${indexName}`
        });
        return true;
      } catch (error) {
        if (error.meta.statusCode === 404) {
          return false;
        } else {
          this.logErrorDetails(error, `Fail to verify index "${indexName}"`);
        }
      }
      return false;
    });
    (0, _defineProperty2.default)(this, "docExists", async ({
      request,
      indexName,
      docId
    }) => {
      try {
        await this.clusterClient.asScoped(request).asCurrentUser.transport.request({
          method: 'get',
          path: `/${indexName}/_doc/${docId}`
        });
        return true;
      } catch (error) {
        if (error.meta.statusCode === 404) {
          return false;
        } else {
          this.logErrorDetails(error, `Fail to verify doc "${docId}"`);
        }
      }
      return false;
    });
    (0, _defineProperty2.default)(this, "createIndexAlias", async ({
      request,
      aliasName,
      indices
    }) => {
      try {
        return await this.clusterClient.asScoped(request).asCurrentUser.indices.putAlias({
          index: indices,
          name: aliasName
        });
      } catch (error) {
        this.logErrorDetails(error, `Fail to create alias "${aliasName}" for indices "${indices}"`);
      }
    });
    (0, _defineProperty2.default)(this, "createDoc", async ({
      request,
      tenantName,
      versionIndexName,
      docId,
      docBody,
      refresh = true
    } = {}) => {
      try {
        // Create doc and index
        await this.clusterClient.asScoped(request).asCurrentUser.create({
          id: docId,
          body: docBody,
          index: versionIndexName,
          refresh
        });
      } catch (error) {
        const docExists = (0, _lodash.get)(error, 'meta.statusCode') === 409;
        if (!docExists) {
          this.logErrorDetails(error, `Fail to create the doc "${docId}" for tenant "${tenantName}" in index ${versionIndexName}`);
        }
      }
    });
    (0, _defineProperty2.default)(this, "createIndexAndAliases", async ({
      request,
      tenantName,
      aliasName,
      versionAliasName,
      versionIndexName
    } = {}) => {
      try {
        await this.clusterClient.asScoped(request).asCurrentUser.indices.create({
          index: versionIndexName,
          body: {
            "mappings": this.activeMappings
          }
        });

        // We must create an alias and a version alias. The migration algorithm requires the alias.
        // And the Kibana page is broken after a tenant is selected if there is no version alias because apps query the version alias directly.
        const aliasesToCreate = [aliasName, versionAliasName];
        return Promise.all(aliasesToCreate.map(aliasName => this.createIndexAlias({
          request,
          aliasName,
          indices: [versionIndexName]
        })));
      } catch (error) {
        const indexExists = (0, _lodash.get)(error, 'meta.statusCode') === 400 && (0, _lodash.get)(error, 'meta.body.error.type') === 'resource_already_exists_exception';
        if (!indexExists) {
          this.logErrorDetails(error, `Fail to create the index "${versionIndexName}" for tenant "${tenantName}"`);
        }
      }
    });
    (0, _defineProperty2.default)(this, "createIndexForTenant", async ({
      request,
      selectedTenant = ''
    } = {}) => {
      const indexExists = await this.indexExists({
        request,
        indexName: this.aliasName
      });
      if (!indexExists) {
        return this.createIndexAndAliases({
          request,
          aliasName: this.aliasName,
          versionAliasName: this.versionAliasName,
          versionIndexName: this.versionIndexName,
          tenantName: selectedTenant
        });
      }
    });
    this.kibanaVersion = coreContext.env.packageInfo.version;
    this.clusterClient = clusterClient;
    this.logger = logger;
    this.configService = configService;

    /*
      The SG backend maps calls for the alias .kibana and the version alias .kibana_<V> to the selectedTenant aliases.
      For example:
        .kibana -> .kibana_-<N>_tenant
        .kibana_V -> .kibana_-<N>_tenant_<V>
    */
    this.aliasName = this.configService.get('kibana.index');
    this.versionAliasName = this.aliasName + `_${this.kibanaVersion}`;
    this.versionIndexName = this.versionAliasName + '_001';

    // we need the active mappings in order to apply it to newly created tenant indices
    const {
      migratorDeps
    } = (0, _tenants_migration_service.setupMigratorDependencies)({
      configService,
      esClient: clusterClient,
      savedObjects,
      kibanaVersion: this.kibanaVersion,
      logger: this.logger
    });
    this.activeMappings = new _migrations.KibanaMigrator(migratorDeps).getActiveMappings();
  }
}
exports.TenantService = TenantService;
//# sourceMappingURL=data:application/json;charset=utf-8;base64,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