"use strict";

Object.defineProperty(exports, "__esModule", {
  value: true
});
exports.TenantsMigrationService = void 0;
exports.setupMigratorDependencies = setupMigratorDependencies;
var _client = require("../../../../../src/core/server/elasticsearch/client");
var _migrations = require("../../../../../src/core/server/saved_objects/migrations");
var _routes = require("./routes");
var _configSchema = require("@kbn/config-schema");
var _default_config = require("../../default_config");
/*
 *    Copyright 2021 floragunn GmbH
 *
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 * http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 */

function setupMigratorDependencies({
  configService,
  esClient,
  savedObjects,
  kibanaVersion,
  logger
}) {
  const savedObjectsMigrationConfig = configService.get('searchguard.multitenancy.saved_objects_migration');
  const soMigrationsConfig = {
    batchSize: savedObjectsMigrationConfig.batch_size,
    scrollDuration: savedObjectsMigrationConfig.scroll_duration,
    pollInterval: savedObjectsMigrationConfig.poll_interval,
    skip: savedObjectsMigrationConfig.skip,
    enableV2: savedObjectsMigrationConfig.enableV2,
    maxBatchSizeBytes: savedObjectsMigrationConfig.max_batch_size ? _configSchema.ByteSizeValue.parse(savedObjectsMigrationConfig.max_batch_size) : _configSchema.ByteSizeValue.parse(_default_config.DEFAULT_CONFIG.multitenancy.saved_objects_migration.max_batch_size)
  };
  const typeRegistry = savedObjects.getTypeRegistry();
  const kibanaConfig = configService.get('kibana');
  const migratorDeps = {
    client: esClient.asInternalUser,
    kibanaConfig,
    typeRegistry,
    logger,
    kibanaVersion,
    soMigrationsConfig
  };
  return {
    soMigrationsConfig,
    typeRegistry,
    kibanaConfig,
    migratorDeps
  };
}
class TenantsMigrationService {
  constructor(coreContext) {
    this.coreContext = coreContext;
    this.kibanaVersion = this.coreContext.env.packageInfo.version;
    this.logger = coreContext.logger.get('tenants-migration-service');
  }
  async setup({
    configService,
    savedObjects,
    esClient,
    kibanaRouter,
    searchGuardBackend
  }) {
    const {
      migratorDeps
    } = setupMigratorDependencies({
      configService,
      esClient,
      savedObjects,
      kibanaVersion: this.kibanaVersion,
      logger: this.logger
    });
    (0, _routes.defineMigrateRoutes)({
      KibanaMigrator: _migrations.KibanaMigrator,
      migratorDeps,
      kibanaRouter,
      searchGuardBackend
    });
  }
  async start({
    searchGuardBackend,
    esClient,
    configService,
    savedObjects
  }) {
    this.logger.debug('Start tenants saved objects migration');
    try {
      const {
        soMigrationsConfig,
        migratorDeps,
        kibanaConfig
      } = setupMigratorDependencies({
        configService,
        esClient,
        savedObjects,
        kibanaVersion: this.kibanaVersion,
        logger: this.logger
      });
      const tenantIndices = await searchGuardBackend.getTenantInfoWithInternalUser();
      this.tenantIndices = !tenantIndices || typeof tenantIndices !== 'object' ? [] : Object.keys(tenantIndices);
      const migrator = new _migrations.KibanaMigrator(migratorDeps);
      const isMigrationNeeded = soMigrationsConfig.skip || !!this.tenantIndices.length;
      if (!isMigrationNeeded) {
        if (soMigrationsConfig.skip) {
          this.logger.info('You configured to skip tenants saved objects migration.');
        } else {
          this.logger.info('No tenants indices found. Thus there is no need for the tenants saved objects migration.');
        }
        return;
      }
      this.logger.info('Starting tenants saved objects migration ...');
      for (let i = 0; i < this.tenantIndices.length; i++) {
        let response;
        try {
          // We execute the index migration sequentially because Elasticsearch doesn't keep up
          // with parallel migration for a large number of indices (tenants).
          // https://git.floragunn.com/search-guard/search-guard-kibana-plugin/-/issues/315
          const kibanaMigrator = new _migrations.KibanaMigrator({
            ...migratorDeps,
            kibanaConfig: {
              ...kibanaConfig,
              index: this.tenantIndices[i]
            }
          });
          kibanaMigrator.prepareMigrations();
          await kibanaMigrator.runMigrations();
          this.logger.info(`Fulfilled migration for index ${this.tenantIndices[i]}.`);
          this.logger.debug(`Migration result:\n${JSON.stringify(response, null, 2)}`);
        } catch (error) {
          this.logger.error(`Unable to fulfill migration for index ${this.tenantIndices[i]}, ${error}`, error);
        }
      }
    } catch (error) {
      throw error;
    }
  }
}
exports.TenantsMigrationService = TenantsMigrationService;
//# sourceMappingURL=data:application/json;charset=utf-8;base64,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