"use strict";

var _interopRequireDefault = require("@babel/runtime/helpers/interopRequireDefault");
Object.defineProperty(exports, "__esModule", {
  value: true
});
exports.AuthManager = exports.AUTH_TYPE_NAMES = void 0;
var _defineProperty2 = _interopRequireDefault(require("@babel/runtime/helpers/defineProperty"));
var _router = require("../../../../../../src/core/server/http/router");
var _lodash = require("lodash");
var _path = _interopRequireDefault(require("path"));
/*
 *    Copyright 2021 floragunn GmbH
 *
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 * http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 */

const AUTH_TYPE_NAMES = exports.AUTH_TYPE_NAMES = {
  BASIC: 'basicauth',
  OIDC: 'oidc',
  JWT: 'jwt',
  SAML: 'saml'
};
class AuthManager {
  constructor({
    kibanaCore,
    sessionStorageFactory,
    pluginDependencies,
    logger,
    searchGuardBackend,
    configService,
    spacesService
  }) {
    /**
     * This needs to be the very first onPreAuth handler that
     * we register for the plugin
     * @param request
     * @param response
     * @param toolkit
     * @returns {Promise<*>}
     */
    (0, _defineProperty2.default)(this, "onPreAuth", async (request, response, toolkit) => {
      if (request.headers.authorization) {
        const sessionCookie = (await this.sessionStorageFactory.asScoped(request).get()) || {};
        const authInstance = await this.getAuthInstanceByCookie({
          request
        });
        if (sessionCookie.credentials && authInstance) {
          // In case we already had a session BEFORE we encountered a request
          // with auth headers, we may need to clear the cookie.
          // Make sure to clear any auth related cookie info if we detect a different header
          await authInstance.clear(request, true);
        }
      }
      return toolkit.next();
    });
    (0, _defineProperty2.default)(this, "checkAuth", async (request, response, toolkit) => {
      const sessionCookie = (await this.sessionStorageFactory.asScoped(request).get()) || {};
      if (request.headers.authorization) {
        return toolkit.authenticated({
          requestHeaders: {
            authorization: request.headers.authorization
          }
        });
      }
      if (this.routesToIgnore.includes(request.url.pathname)) {
        // Change back after everything has been implemented
        return toolkit.notHandled();
      }
      if (this.unauthenticatedRoutes.includes(request.url.pathname)) {
        // If we do this, we don't really assign any relevant headers
        // Until now, we got the kibana server user here, but those credentials were
        // not really used, it seems
        return toolkit.authenticated({});
      }
      const authInstanceByRequest = await this.getAuthInstanceByRequest({
        request
      });
      if (authInstanceByRequest) {
        return authInstanceByRequest.checkAuth(request, response, toolkit);
      }

      // @todo This way of handling anonymous auth unfortunately
      // doesn't provide a good way of showing an error message
      // if the SG backend hasn't been configured properly
      if (!sessionCookie.authType && this.configService.get('searchguard.auth.anonymous_auth_enabled')) {
        return toolkit.authenticated({});
      }
      const isAjaxRequest = request.headers && (request.headers.accept && request.headers.accept.split(',').indexOf('application/json') > -1 || request.headers['content-type'] && request.headers['content-type'].indexOf('application/json') > -1);
      const nextUrl = this.getNextUrl(request);
      let loginPageURL = this.basePath + '/login' + `?nextUrl=${nextUrl}`;
      try {
        const authConfig = await this.searchGuardBackend.getAuthConfig(nextUrl);
        let config;
        if (authConfig && authConfig.auth_methods && authConfig.auth_methods.length == 1 && authConfig.auth_methods[0].sso_location) {
          // If there is only one auth_method with sso_location
          config = authConfig.auth_methods[0];
        } else {
          // If one of the methods has auto_select property enabled
          config = authConfig && authConfig.auth_methods && authConfig.auth_methods.find(({
            auto_select
          }) => auto_select);
        }
        if (config && config.sso_location) {
          loginPageURL = config.sso_location;
          const authInstance = this.authInstances[config.method];
          if (authInstance && authInstance.loginURL) {
            loginPageURL = new URL(this.basePath + authInstance.loginURL, 'http://abc');
            if (config.id) {
              loginPageURL.searchParams.set('authTypeId', config.id);
            }
            if (nextUrl) {
              loginPageURL.searchParams.set('nextUrl', nextUrl);
            }
            loginPageURL = loginPageURL.href.replace(loginPageURL.origin, '');
          }
          if (config.capture_url_fragment && nextUrl && !isAjaxRequest) {
            return response.redirected({
              headers: {
                'location': `${this.basePath}/auth/captureurlfragment?loginHandler=${this.basePath + authInstance.loginURL}&authTypeId=${config.id}&nextUrl=${encodeURIComponent(nextUrl)}`
              }
            });
          }
        }
      } catch (error) {
        console.error("Error while retrieving auth config", error);
      }
      if (isAjaxRequest) {
        // If the session has expired, we may receive ajax requests that can't handle a 302 redirect.
        // In this case, we trigger a 401 and let the interceptor handle the redirect on the client side.
        return response.unauthorized({
          headers: {
            sg_redirectTo: loginPageURL
          },
          body: {
            message: 'Session expired or invalid username and password'
          }
        });
      }
      return response.redirected({
        headers: {
          location: loginPageURL
        }
      });
    });
    // @todo Not needed for 7.10?
    (0, _defineProperty2.default)(this, "onPostAuth", async (request, response, toolkit) => {
      if (request.route.path === '/api/core/capabilities') {
        if (this.configService.get('searchguard.auth.anonymous_auth_enabled')) return toolkit.next();
        const authHeaders = await this.getAllAuthHeaders(request);
        if (authHeaders === false && !request.headers.authorization) {
          /*
          We need this redirect because Kibana calls the capabilities on our login page. The Kibana checks if there is the default space in the Kibana index.
          The problem is that the Kibana call is scoped to the current request. And the current request doesn't contain any credentials in the headers because the user hasn't been authenticated yet.
          As a result, the call fails with 401, and the user sees the Kibana error page instead of our login page.
          We flank this issue by redirecting the Kibana call to our route /api/v1/searchguard/kibana_capabilities where we serve some
          minimum amount of capabilities. We expect that Kibana fetches the capabilities again once the user logged in.
          */
          // The payload is passed together with the redirect despite of the undefined here
          return new _router.KibanaResponse(307, undefined, {
            headers: {
              location: this.basePath + '/api/v1/searchguard/kibana_capabilities'
            }
          });
        }
      }
      return toolkit.next();
    });
    this.kibanaCore = kibanaCore;
    this.sessionStorageFactory = sessionStorageFactory;
    this.searchGuardBackend = searchGuardBackend;
    this.logger = logger;
    this.pluginDependencies = pluginDependencies;
    this.configService = configService;
    this.spacesService = spacesService;
    this.authInstances = {};
    this.unauthenticatedRoutes = this.configService.get('searchguard.auth.unauthenticated_routes');

    /**
     * Loading bundles are now behind auth.
     * We need to skip auth for the bundles for the login page and the error page
     */
    this.routesToIgnore = [
    //'/login',
    '/customerror', '/bootstrap-anonymous.js', '/bundles/app/core/bootstrap.js', '/bundles/app/searchguard-customerror/bootstrap.js',
    // SAML specific
    '/searchguard/saml/acs', '/searchguard/saml/acs/idpinitiated', '/searchguard/saml/logout'];
    this.basePath = kibanaCore.http.basePath.get();
  }
  registerAuthInstances() {
    [require('./types/openid/OpenId'), require('./types/basicauth/BasicAuth'), require('./types/jwt/Jwt'), require('./types/saml/Saml')].forEach(AuthClass => {
      // @todo This needs to respect the order as given by the backend
      const authInstance = new AuthClass({
        kibanaCore: this.kibanaCore,
        sessionStorageFactory: this.sessionStorageFactory,
        pluginDependencies: this.pluginDependencies,
        logger: this.logger,
        searchGuardBackend: this.searchGuardBackend,
        config: this.configService,
        authManager: this,
        // @todo Is the authManager used?
        spacesService: this.spacesService
      });
      authInstance.init();
      this.authInstances[authInstance.type] = authInstance;
    });
  }
  registerAuthInstance(authTypeName, authInstance) {
    this.authInstances[authTypeName] = authInstance;
  }
  getAuthInstanceByName(authTypeName) {
    if (this.authInstances[authTypeName]) {
      return this.authInstances[authTypeName];
    }
    return null;
  }
  async getAuthInstanceByRequest({
    request
  }) {
    const matchedAuthInstance = await this.getAuthInstanceByAuthTypes({
      request
    });
    // matchedAuthInstance will be null if we didn't get a match
    if (matchedAuthInstance) {
      return matchedAuthInstance;
    }
    const authInstanceByCookie = await this.getAuthInstanceByCookie({
      request
    });
    if (authInstanceByCookie) {
      return authInstanceByCookie;
    }
    return null;
  }
  async getAuthInstanceByAuthTypes({
    request
  }) {
    for (const authType in this.authInstances) {
      const authInstance = this.getAuthInstanceByName(authType);
      const authInstanceResult = await authInstance.detectCredentialsByRequest({
        request
      });
      if (authInstanceResult !== null) {
        return authInstance;
      }
    }
    return null;
  }
  async getAuthInstanceByCookie({
    request
  }) {
    const sessionCookie = (await this.sessionStorageFactory.asScoped(request).get()) || {};
    if (sessionCookie.authType && this.authInstances[sessionCookie.authType]) {
      return this.getAuthInstanceByName(sessionCookie.authType);
    }
    return null;
  }
  getNextUrl(request) {
    let nextUrl = _path.default.posix.join(this.basePath, request.url.pathname);
    if (request.url.search) nextUrl += request.url.search;
    return nextUrl;
  }

  /**
   * Get credentials from an existing cookie only
   * @param request
   * @returns {Promise<*|boolean|boolean>}
   */
  async getAuthHeader(request) {
    const authInstance = await this.getAuthInstanceByCookie({
      request
    });
    if (authInstance) {
      // @todo A bit weird that we have different method signatures here
      const sessionCookie = (await this.sessionStorageFactory.asScoped(request).get()) || {};
      return authInstance.getAuthHeader(sessionCookie);
    }
    return false;
  }
  async getAllAuthHeaders(request) {
    if (request.headers.authorization) {
      return false;
    }
    const authInstance = await this.getAuthInstanceByRequest({
      request
    });
    if (authInstance) {
      return authInstance.getAllAuthHeaders(request);
    }
    return false;
  }
  async logout({
    context,
    request,
    response
  }) {
    const authInstance = await this.getAuthInstanceByCookie({
      request
    });
    if (authInstance) {
      return await authInstance.logout({
        context,
        request,
        response
      });
    }
    return response.ok();
  }
}
exports.AuthManager = AuthManager;
//# sourceMappingURL=data:application/json;charset=utf-8;base64,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