"use strict";

var _interopRequireDefault = require("@babel/runtime/helpers/interopRequireDefault");
Object.defineProperty(exports, "__esModule", {
  value: true
});
exports.default = void 0;
var _defineProperty2 = _interopRequireDefault(require("@babel/runtime/helpers/defineProperty"));
var _lodash = require("lodash");
var _response = require("../../../../../../../src/core/server/http/router/response");
var _router = require("../../../../../../../src/core/server/http/router");
var _invalid_session_error = _interopRequireDefault(require("../errors/invalid_session_error"));
var _session_expired_error = _interopRequireDefault(require("../errors/session_expired_error"));
var _filter_auth_headers = _interopRequireDefault(require("../filter_auth_headers"));
var _missing_tenant_error = _interopRequireDefault(require("../errors/missing_tenant_error"));
var _path = _interopRequireDefault(require("path"));
/* eslint-disable @kbn/eslint/require-license-header */
/**
 *    Copyright 2018 floragunn GmbH

 Licensed under the Apache License, Version 2.0 (the "License");
 you may not use this file except in compliance with the License.
 You may obtain a copy of the License at

 http://www.apache.org/licenses/LICENSE-2.0

 Unless required by applicable law or agreed to in writing, software
 distributed under the License is distributed on an "AS IS" BASIS,
 WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 See the License for the specific language governing permissions and
 limitations under the License.
 */

class AuthType {
  constructor({
    searchGuardBackend,
    kibanaCore,
    config,
    logger,
    sessionStorageFactory,
    pluginDependencies,
    spacesService
  }) {
    /**
     * Called internally and checks for an AJAX request before
     * invoking the auth type's OnUnAuthenticated method.
     * @param request
     * @param response
     * @param toolkit
     * @param error
     * @returns {Promise<void|*>}
     * @private
     */
    (0, _defineProperty2.default)(this, "_handleUnAuthenticated", async (request, response, toolkit, error = null) => {
      if (request.headers) {
        // If the session has expired, we may receive ajax requests that can't handle a 302 redirect.
        // In this case, we trigger a 401 and let the interceptor handle the redirect on the client side.
        if (request.headers.accept && request.headers.accept.split(',').indexOf('application/json') > -1 || request.headers['content-type'] && request.headers['content-type'].indexOf('application/json') > -1) {
          this.debugLog('Not authenticated, detected AJAX request');
          const sessionCookie = (await this.sessionStorageFactory.asScoped(request).get()) || {};
          if (request.route.path === '/api/core/capabilities') {
            return toolkit.notHandled();
          }
          return response.unauthorized({
            headers: {
              sg_redirectTo: await this.getRedirectTargetForUnauthenticated(request, error, true, sessionCookie)
            },
            body: {
              message: 'Session expired'
            }
          });
        }
      }
      return this.onUnAuthenticated(request, response, toolkit, error);
    });
    (0, _defineProperty2.default)(this, "onPostAuth", async (request, response, toolkit) => {
      if (request.route.path === '/api/core/capabilities') {
        const sessionCookie = (await this.sessionStorageFactory.asScoped(request).get()) || {};
        if (sessionCookie.isAnonymousAuth) return toolkit.next();
        const authHeaders = await this.getAllAuthHeaders(request);
        if (authHeaders === false) {
          /*
          We need this redirect because Kibana calls the capabilities on our login page. The Kibana checks if there is the default space in the Kibana index.
          The problem is that the Kibana call is scoped to the current request. And the current request doesn't contain any credentials in the headers because the user hasn't been authenticated yet.
          As a result, the call fails with 401, and the user sees the Kibana error page instead of our login page.
          We flank this issue by redirecting the Kibana call to our route /api/v1/searchguard/kibana_capabilities where we serve some
          minimum amount of capabilities. We expect that Kibana fetches the capabilities again once the user logged in.
          */
          // The payload is passed together with the redirect despite of the undefined here
          return new _response.KibanaResponse(307, undefined, {
            headers: {
              location: this.basePath + '/api/v1/searchguard/kibana_capabilities'
            }
          });
        } else {
          // Update the request with auth headers in order to allow Kibana to check the default space.
          // Kibana page breaks if Kibana can't check the default space.
          const rawRequest = (0, _router.ensureRawRequest)(request);
          (0, _lodash.assign)(rawRequest.headers, authHeaders);
        }
      }
      return toolkit.next();
    });
    (0, _defineProperty2.default)(this, "checkAuth", async (request, response, toolkit) => {
      let sessionCookie = (await this.sessionStorageFactory.asScoped(request).get()) || {};
      try {
        sessionCookie = await this.getCookieWithCredentials(request);
      } catch (error) {
        return this._handleUnAuthenticated(request, response, toolkit, error);
      }
      if (sessionCookie.credentials) {
        const authHeaders = await this.getAllAuthHeaders(request, sessionCookie);
        if (!authHeaders) {
          this.logger.error(`An error occurred while computing auth headers, clearing session: No headers found in the session cookie`);
          await this.clear(request);
          return this._handleUnAuthenticated(request, response, toolkit);
        }
        const isMtEnabled = this.config.get('searchguard.multitenancy.enabled');
        if (!isMtEnabled && this.pluginDependencies.spaces) {
          await this.spacesService.createDefaultSpace({
            request: {
              headers: authHeaders
            }
          });
        }
        return toolkit.authenticated({
          requestHeaders: authHeaders
        });
      }
      return this._handleUnAuthenticated(request, response, toolkit);
    });
    this.searchGuardBackend = searchGuardBackend;
    this.config = config;
    this.kibanaCore = kibanaCore;
    this.logger = logger;
    this.sessionStorageFactory = sessionStorageFactory;
    this.pluginDependencies = pluginDependencies;
    this.spacesService = spacesService;
    this.basePath = kibanaCore.http.basePath.get();
    this.frontendBaseUrl = this.config.get('searchguard.frontend_base_url') || kibanaCore.http.basePath.publicBaseUrl;
    this.sgFrontendConfigId = this.config.get('searchguard.sg_frontend_config_id') || 'default';
    if (!this.frontendBaseUrl) {
      const serverInfo = kibanaCore.http.getServerInfo();
      this.frontendBaseUrl = serverInfo.protocol + '://' + serverInfo.hostname + ':' + serverInfo.port + '/' + kibanaCore.http.basePath.serverBasePath;
    }
    this.authDebugEnabled = this.config.get('searchguard.auth.debug');

    /**
     * The authType is saved in the auth cookie for later reference
     * @type {string}
     */
    this.type = null;

    /**
     * If a loginURL is defined, we can skip the auth selector page
     * if the customer only has one auth type enabled.
     * @type {string|null}
     */
    this.loginURL = null;

    /**
     * Tells the sessionPlugin whether or not to validate the number of tenants when authenticating
     * @type {boolean}
     */
    this.validateAvailableTenants = true;

    /**
     * The name of the header were we look for an authorization value.
     * This should most likely be set in the subclass depending on a config value.
     * @type {string}
     */
    this.authHeaderName = 'authorization';

    /**
     * Additional headers that should be passed as part as the authentication.
     * Do not use headers here that have an effect on which user is logged in.
     * @type {string[]}
     */
    this.allowedAdditionalAuthHeaders = ['sg_impersonate_as'];
  }
  async init() {
    this.setupRoutes();
  }

  /**
   * Returns the auth header needed for the Search Guard backend
   * @param session
   * @returns {*}
   */
  getAuthHeader(session) {
    if (session.credentials && session.credentials.authHeaderValue) {
      return {
        [this.authHeaderName]: session.credentials.authHeaderValue
      };
    }
    return false;
  }

  /**
   * Can be used by auth types that need to handle cases
   * where the credentials are passed together with the
   * request.
   * Example: JWT supports passing the bearer token per query parameter
   *
   * NB: Should NOT be used to detect pre-authenticated requests.
   * For those, we don't want to create a cookie.
   *
   * @param request
   * @returns {Promise<null>}
   */
  async detectCredentialsByRequest({
    request
  }) {
    return null;
  }

  /**
   * Checks if we have an authorization header.
   *
   * Pass the existing session credentials to compare with the authorization header.
   *
   * @param request
   * @param sessionCredentials
   * @returns {object|null} - credentials for the authentication
   */
  detectAuthHeaderCredentials(request, sessionCredentials = null) {
    if (request.headers[this.authHeaderName]) {
      const authHeaderValue = request.headers[this.authHeaderName];

      // If we have sessionCredentials AND auth headers we need to check if they are the same.
      if (sessionCredentials !== null && sessionCredentials.authHeaderValue === authHeaderValue) {
        // The auth header credentials are the same as those in the session,
        // no need to return new credentials so we're just returning null here
        return null;
      }
      return {
        authHeaderValue: authHeaderValue
      };
    }
    return null;
  }
  async getRedirectTargetForUnauthenticated() {
    throw new Error('The getRedirectTargetForUnauthenticated method must be implemented by the sub class');
  }
  async authenticate(credentials, options = {}, additionalAuthHeaders = {}) {
    try {
      this.debugLog('Authenticating using ' + credentials);
      credentials.frontend_base_url = this.frontendBaseUrl;
      credentials.config_id = this.sgFrontendConfigId;
      const sessionResponse = await this.searchGuardBackend.authenticateWithSession(credentials);
      const sessionCredentials = {
        authHeaderValue: 'Bearer ' + sessionResponse.token
      };
      this.debugLog('Token ' + sessionCredentials.authHeaderValue);
      const user = await this.searchGuardBackend.authenticateWithHeader(this.authHeaderName, sessionCredentials.authHeaderValue, additionalAuthHeaders);
      const session = {
        username: user.username,
        // The session token
        credentials: sessionCredentials,
        authType: this.type,
        authTypeId: credentials.id
      };
      return {
        session,
        user,
        redirectUri: sessionResponse.redirect_uri
      };
    } catch (error) {
      throw error;
    }
  }
  async onUnAuthenticated(request, response, toolkit, error = null) {
    const redirectTo = await this.getRedirectTargetForUnauthenticated(request, error);
    return response.redirected({
      headers: {
        location: `${redirectTo}`
      }
    });
  }

  /**
   * A helper for generating the correct nextUrl.
   * Spaces manipulates the URL for non default
   * spaces, and that change is not reflected
   * in request.url.pathname
   * @param request
   * @returns {string}
   */
  getNextUrl(request) {
    return _path.default.posix.join(this.basePath, request.url.pathname) + request.url.search;
  }
  setupRoutes() {
    throw new Error('The getAuthHeader method must be implemented by the sub class');
  }
  async getCookieWithCredentials(request) {
    const authHeaderCredentials = await this.detectCredentialsByRequest({
      request
    });
    if (authHeaderCredentials) {
      try {
        this.debugLog('Got auth header credentials, trying to authenticate');
        const {
          session
        } = await this.handleAuthenticate(request, authHeaderCredentials);
        return session;
      } catch (error) {
        this.logger.error(`Got auth header credentials, but authentication failed: ${error.stack}`);
        // Fall through
      }
    }
    let sessionCookie = (await this.sessionStorageFactory.asScoped(request).get()) || {};
    if (sessionCookie.credentials) {
      try {
        return await this.validateSessionCookie(request, sessionCookie);
      } catch (error) {
        // We can return early here. Even if we have valid request headers,
        // the cookie would have been updated in the validator.
        // Logging this as info since it isn't really an error, but just a part of the flow.
        this.logger.info(`Got credentials, but the validation failed: ${error.stack}`);
        // Fall through
      }
    }

    // No (valid) cookie, we need to check for headers

    return sessionCookie;
  }
  /**
   * If a session auth cookie exists, the sessionValidator is called to validate the content.
   * If the cookie isn't valid, an error will be thrown
   * @param server
   * @returns {validate}
   */
  async validateSessionCookie(request, sessionCookie) {
    if (sessionCookie.authType !== this.type) {
      await this.clear(request);
      throw new _invalid_session_error.default('Invalid cookie');
    }
    const checkTokenExpirationTime = Date.now();
    if (!sessionCookie.checkTokenExpirationTime || checkTokenExpirationTime - sessionCookie.checkTokenExpirationTime > 15000) {
      try {
        const authHeader = this.getAuthHeader(sessionCookie);
        const authInfoResponse = await this.searchGuardBackend.authinfo(authHeader || request.headers);
        sessionCookie.checkTokenExpirationTime = checkTokenExpirationTime;
        await this.sessionStorageFactory.asScoped(request).set(sessionCookie);
        if (authInfoResponse.user_name !== sessionCookie.username) {
          throw new Error('We have a different user in the cookie. Most likely because of anonymous auth.');
        }
      } catch (error) {
        await this.clear(request);
        throw new _session_expired_error.default('Session expired');
      }
    }
    return sessionCookie;
  }

  /**
   * Get all auth headers based on the current request.
   *
   * @param request
   * @param sessionCookie
   * @returns {Promise<boolean|*>}
   */
  async getAllAuthHeaders(request, sessionCookie = null) {
    if (!sessionCookie) {
      try {
        sessionCookie = await this.getCookieWithCredentials(request);
      } catch (error) {
        this.logger.info(`Getting all auth headers failed: ${error.stack}`);
      }
    }
    if (!sessionCookie || !sessionCookie.credentials) {
      return false;
    }
    const authHeader = this.getAuthHeader(sessionCookie);
    if (authHeader !== false) {
      this.addAdditionalAuthHeaders(request, authHeader, sessionCookie);
      return authHeader;
    }
    return false;
  }

  /**
   * @deprecated
   *
   * Method for adding additional auth type specific authentication headers.
   * Override this in the auth type for type specific headers.
   *
   * NB: Remember to call the super method if you do.
   *
   * @param request
   * @param authHeader
   * @param session
   */
  addAdditionalAuthHeaders(request, authHeader, session) {
    if (session.additionalAuthHeaders) {
      for (const header in session.additionalAuthHeaders) {
        if (session.additionalAuthHeaders.hasOwnProperty(header)) {
          authHeader[header] = session.additionalAuthHeaders[header];
        }
      }
    }
  }
  debugLog(message, label = this.type) {
    if (this.authDebugEnabled) {
      try {
        if (typeof message !== 'string') {
          message = JSON.stringify(message);
        }
        this.logger.info(`${label} ${message}`);
      } catch (error) {
        this.logger.error(`Error in debug log: ${error.stack}`);
      }
    }
  }

  /**
   * Tries to authenticate a user. If multitenancy is enabled, we also try to validate that the
   * user has at least one valid tenant
   * @param {object} request
   * @param {object} headers
   * @param {object} credentials
   * @returns {Promise<*>}
   */
  async handleAuthenticate(request, credentials, options = {}) {
    try {
      const additionalAuthHeaders = (0, _filter_auth_headers.default)(request.headers, this.allowedAdditionalAuthHeaders);
      // authResponse is an object with .session and .user
      const authResponse = await this.authenticate(credentials, options, additionalAuthHeaders);
      return this._handleAuthResponse(request, credentials, authResponse, additionalAuthHeaders);
    } catch (error) {
      // Make sure we clear any existing cookies if something went wrong
      this.clear(request, true);
      throw error;
    }
  }

  /**
   * Normalized response after an authentication
   * @param credentials
   * @param authResponse
   * @returns {*}
   * @private
   */
  async _handleAuthResponse(request, credentials, authResponse, additionalAuthHeaders = {}) {
    // Make sure the user has a tenant that they can use
    if (this.validateAvailableTenants && this.config.get('searchguard.multitenancy.enabled') && !this.config.get('searchguard.multitenancy.tenants.enable_global')) {
      const privateTenantEnabled = this.config.get('searchguard.multitenancy.tenants.enable_private');
      const allTenants = authResponse.user.tenants;
      if (allTenants != null) {
        // We're only here if the global tenant is not enabled,
        // so we need to filter out the SGS_GLOBAL_TENANT too
        delete allTenants.SGS_GLOBAL_TENANT;
      }
      if (allTenants != null && !privateTenantEnabled) {
        delete allTenants[authResponse.user.username];
      }
      if (allTenants == null || Object.keys(allTenants).length === 0) {
        throw new _missing_tenant_error.default('No tenant available for this user, please contact your system administrator.');
      }
    }

    // If we used any additional auth headers when authenticating, we need to store them in the session
    /* @todo Was used with sg_impersonate_as
    authResponse.session.additionalAuthHeaders = null;
    if (Object.keys(additionalAuthHeaders).length) {
      authResponse.session.additionalAuthHeaders = additionalAuthHeaders;
    }
      */

    const cookie = (await this.sessionStorageFactory.asScoped(request).get()) || {};
    authResponse.session = {
      ...cookie,
      ...authResponse.session
    };
    await this.sessionStorageFactory.asScoped(request).set(authResponse.session);
    return authResponse;
  }
  async logout({
    request,
    response
  }) {
    await this.clear(request, true);
    return response.ok({
      body: {
        authType: this.type,
        // @todo Use the kibana_url from the config?
        redirectURL: this.basePath + '/login?type=' + this.type + 'Logout'
      }
    });
  }

  /**
   * Remove the credentials from the session cookie
   */
  async clear(request, explicitlyRemoveAuthType = false) {
    const sessionCookie = (await this.sessionStorageFactory.asScoped(request).get()) || {};
    const authHeaders = this.getAuthHeader(sessionCookie);
    // @todo Consider refactoring anything auth related into an "auth" property.
    delete sessionCookie.credentials;
    delete sessionCookie.username;
    if (explicitlyRemoveAuthType) {
      delete sessionCookie.authType;
      delete sessionCookie.authTypeId;
    }
    delete sessionCookie.additionalAuthHeaders;

    // Only try to delete the session if we really have auth headers
    if (authHeaders) {
      try {
        await this.searchGuardBackend.logoutSession(authHeaders);
      } catch (error) {
        this.logger.error(`Failed to delete the session token: ${error.stack}`);
      }
    }
    return await this.sessionStorageFactory.asScoped(request).set(sessionCookie);
  }
}
exports.default = AuthType;
module.exports = exports.default;
//# sourceMappingURL=data:application/json;charset=utf-8;base64,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