"use strict";

var _interopRequireDefault = require("@babel/runtime/helpers/interopRequireDefault");
Object.defineProperty(exports, "__esModule", {
  value: true
});
exports.WWW_AUTHENTICATE_HEADER_NAME = exports.Kerberos = void 0;
var _defineProperty2 = _interopRequireDefault(require("@babel/runtime/helpers/defineProperty"));
var _response = require("../../../../../../../../src/core/server/http/router/response");
/*
 *    Copyright 2020 floragunn GmbH
 *
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 * http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 */

const WWW_AUTHENTICATE_HEADER_NAME = exports.WWW_AUTHENTICATE_HEADER_NAME = 'WWW-Authenticate';
class Kerberos {
  constructor(props) {
    (0, _defineProperty2.default)(this, "checkAuth", async (request, response, toolkit) => {
      const sessionCookie = (await this.sessionStorageFactory.asScoped(request).get()) || {};
      if (sessionCookie.credentials && sessionCookie.credentials.authHeaderValue) {
        const headers = {
          authorization: sessionCookie.credentials.authHeaderValue
        };
        try {
          const authInfoResponse = await this.searchGuardBackend.authinfo(headers);
        } catch (e) {
          return this.authenticateWithSPNEGO(request, response, toolkit, sessionCookie);
        }
        return toolkit.authenticated({
          requestHeaders: headers
        });
      }
      return this.authenticateWithSPNEGO(request, response, toolkit, sessionCookie);
    });
    this.type = 'kerberos';
    this.logger = props.logger;
    this.config = props.config;
    this.searchGuardBackend = props.searchGuardBackend;
    this.authDebugEnabled = this.config.get('searchguard.auth.debug');
    this.basePath = props.basePath ? props.basePath : "/";
    this.sessionStorageFactory = props.sessionStorageFactory;
  }

  // See the Negotiate Operation Example for the authentication flow details
  // https://tools.ietf.org/html/rfc4559#section-5

  async authenticateWithSPNEGO(request, response, toolkit, sessionCookie) {
    let backendError;
    try {
      const whitelistRoutes = this.config.get('searchguard.auth.unauthenticated_routes');
      if (whitelistRoutes.includes(request.route.path)) {
        return toolkit.authenticated();
      }
      const headers = {};
      if (request.headers.authorization) {
        headers.authorization = request.headers.authorization;
      }

      // Validate the request.
      // The headers.authorization may hold SPNEGO GSSAPI token or basic auth credentials.
      const createSessionResponse = await this.searchGuardBackend.createSessionWithHeaders(headers);
      if (this.authDebugEnabled) {
        this.logger.debug(`Authenticated: ${JSON.stringify(createSessionResponse, null, 2)}.`);
      }
      const authHeaders = {
        authorization: 'Bearer ' + createSessionResponse.token
      };
      const user = await this.searchGuardBackend.authenticateWithHeader("authorization", authHeaders.authorization);
      sessionCookie.username = user.username;
      sessionCookie.credentials = {
        authHeaderValue: authHeaders.authorization
      };
      await this.sessionStorageFactory.asScoped(request).set(sessionCookie);
      return toolkit.authenticated({
        requestHeaders: authHeaders
      });
    } catch (error) {
      backendError = error.inner || error;
      if (backendError && backendError.meta && backendError.meta.headers["www-authenticate"]) {
        let authenticateHeader = backendError.meta.headers["www-authenticate"];
        let parts = authenticateHeader.split(/\s*,\s*/);
        for (let negotiationProposal of parts) {
          if (negotiationProposal.startsWith('Negotiate')) {
            return response.unauthorized({
              headers: {
                [WWW_AUTHENTICATE_HEADER_NAME]: negotiationProposal
              }
            });
          }
        }
      }
      return response.unauthorized({
        body: backendError
      });
    }
  }
  async getCookieWithCredentials(request) {
    const sessionCookie = (await this.sessionStorageFactory.asScoped(request).get()) || {};
    if (sessionCookie.credentials && sessionCookie.credentials.authHeaderValue) {
      const headers = {
        authorization: sessionCookie.credentials.authHeaderValue
      };
      try {
        const authInfoResponse = await this.searchGuardBackend.authinfo(headers);
      } catch (e) {
        return this.authenticateWithSPNEGOtoCookie(request, sessionCookie);
      }
      return sessionCookie;
    }
    return this.authenticateWithSPNEGOtoCookie(request, sessionCookie);
  }
  async authenticateWithSPNEGOtoCookie(request, sessionCookie) {
    try {
      const headers = {};
      if (request.headers.authorization) {
        headers.authorization = request.headers.authorization;
      }

      // Validate the request.
      // The headers.authorization may hold SPNEGO GSSAPI token or basic auth credentials.
      const createSessionResponse = await this.searchGuardBackend.createSessionWithHeaders(headers);
      if (this.authDebugEnabled) {
        this.logger.debug(`Authenticated: ${JSON.stringify(createSessionResponse, null, 2)}.`);
      }
      const authHeaders = {
        authorization: 'Bearer ' + createSessionResponse.token
      };
      const user = await this.searchGuardBackend.authenticateWithHeader("authorization", authHeaders.authorization);
      sessionCookie.username = user.username;
      sessionCookie.credentials = {
        authHeaderValue: authHeaders.authorization
      };
      return sessionCookie;
    } catch (error) {
      console.error(error);
      return sessionCookie;
    }
  }
  getAuthHeader(session) {
    if (session.credentials && session.credentials.authHeaderValue) {
      return {
        authorization: session.credentials.authHeaderValue
      };
    }
    return false;
  }
}
exports.Kerberos = Kerberos;
//# sourceMappingURL=data:application/json;charset=utf-8;base64,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