"use strict";

var _interopRequireDefault = require("@babel/runtime/helpers/interopRequireDefault");
Object.defineProperty(exports, "__esModule", {
  value: true
});
exports.default = void 0;
var _AuthType = _interopRequireDefault(require("../AuthType"));
var _missing_tenant_error = _interopRequireDefault(require("../../errors/missing_tenant_error"));
var _authentication_error = _interopRequireDefault(require("../../errors/authentication_error"));
var _missing_role_error = _interopRequireDefault(require("../../errors/missing_role_error"));
var _routes = require("./routes");
var _path = _interopRequireDefault(require("path"));
var _AuthManager = require("../../AuthManager");
/*
 *    Copyright 2021 floragunn GmbH
 *
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 * http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 */

class OpenId extends _AuthType.default {
  constructor({
    searchGuardBackend,
    kibanaCore,
    config,
    logger,
    sessionStorageFactory,
    pluginDependencies,
    spacesService
  }) {
    super({
      searchGuardBackend,
      kibanaCore,
      config,
      logger,
      sessionStorageFactory,
      pluginDependencies,
      spacesService
    });

    /**
     * The authType is saved in the auth cookie for later reference
     * @type {string}
     */
    this.type = _AuthManager.AUTH_TYPE_NAMES.OIDC;

    /**
     * If a loginURL is defined, we can skip the auth selector page
     * if the customer only has one auth type enabled.
     * @type {string|null}
     */
    this.loginURL = _routes.OIDC_ROUTES.LOGIN;
  }
  async getRedirectTargetForUnauthenticated(request, error = null, isAJAX = false, sessionCookie = {}) {
    const url = new URL(request.url.href);
    url.pathname = _path.default.posix.join(this.basePath, '/customerror');

    // If we don't have any tenant we need to show the custom error page
    if (error instanceof _missing_tenant_error.default) {
      url.searchParams.set('type', 'missingTenant');
    } else if (error instanceof _missing_role_error.default) {
      url.searchParams.set('type', 'missingRole');
    } else if (error instanceof _authentication_error.default) {
      url.searchParams.set('type', 'authError');
    } else {
      if (!isAJAX) {
        url.searchParams.set('nextUrl', this.getNextUrl(request));
        // Delete sg_tenant because we have it already as a param in the nextUrl
        url.searchParams.delete('sg_tenant');
      }
      url.searchParams.set('configAuthTypeId', sessionCookie.authTypeId || null);
      url.pathname = _path.default.posix.join(this.basePath, this.loginURL);
    }
    return url.pathname + url.search + url.hash;
  }
  setupRoutes() {
    (0, _routes.defineRoutes)({
      authInstance: this,
      kibanaCore: this.kibanaCore,
      kibanaConfig: this.config,
      logger: this.logger,
      debugLog: this.debugLog.bind(this),
      searchGuardBackend: this.searchGuardBackend
    });
  }

  /**
   * Clears the session and logs the user out from the IdP (if we have an endpoint available)
   * @url http://openid.net/specs/openid-connect-session-1_0.html#RPLogout
   * @param context
   * @param request
   * @param response
   * @returns {Promise<*>}
   */
  async logout({
    context = null,
    request,
    response
  }) {
    // @todo Auth error isn't the best message for this. We still
    // get logged out from Kibana, but the IdP logout may fail.
    let redirectURL = `${this.basePath}/customerror?type=authError`;
    const sessionCookie = (await this.sessionStorageFactory.asScoped(request).get()) || {};
    const authHeader = this.getAuthHeader(sessionCookie);
    try {
      const sessionInfo = await this.searchGuardBackend.sessionInfo(authHeader);
      // sso_logout_url doesn't always exist
      redirectURL = sessionInfo.sso_logout_url || this.basePath + '/login?type=' + this.type + 'Logout';
    } catch (error) {
      this.logger.error(`OIDC auth logout failed while retrieving the sso_logout_url: ${error.stack}`);
    }

    // Clear the cookie credentials
    await this.clear(request, true);
    return response.ok({
      body: {
        authType: this.type,
        redirectURL
      }
    });
  }
}
exports.default = OpenId;
module.exports = exports.default;
//# sourceMappingURL=data:application/json;charset=utf-8;base64,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