"use strict";

Object.defineProperty(exports, "__esModule", {
  value: true
});
exports.SAML_ROUTES = void 0;
exports.defineRoutes = defineRoutes;
var _sanitize_next_url = require("../../sanitize_next_url");
var _configSchema = require("@kbn/config-schema");
var _constants = require("../../../../../utils/constants");
/* eslint-disable @kbn/eslint/require-license-header */
/**
 *    Copyright 2018 floragunn GmbH

 Licensed under the Apache License, Version 2.0 (the "License");
 you may not use this file except in compliance with the License.
 You may obtain a copy of the License at

 http://www.apache.org/licenses/LICENSE-2.0

 Unless required by applicable law or agreed to in writing, software
 distributed under the License is distributed on an "AS IS" BASIS,
 WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 See the License for the specific language governing permissions and
 limitations under the License.
 */

const SAML_ROUTES = exports.SAML_ROUTES = {
  LOGIN: `${_constants.APP_ROOT}/auth/saml/login`
};
function defineRoutes({
  authInstance,
  searchGuardBackend,
  kibanaCore,
  debugLog,
  sessionStorageFactory,
  logger,
  configService
}) {
  const basePath = kibanaCore.http.basePath.serverBasePath;
  const httpResources = kibanaCore.http.resources;
  const router = kibanaCore.http.createRouter();
  const httpAuth = kibanaCore.http.auth;
  const routesPath = '/auth/saml/';

  /**
   * The login page.
   */
  router.get({
    path: `${_constants.APP_ROOT}${routesPath}login`,
    validate: false,
    options: {
      authRequired: false
    }
  }, async function (context, request, response) {
    const sessionCookie = (await sessionStorageFactory.asScoped(request).get()) || {};
    /*
    When successful logout,
      headers: {
        cookie: 'searchguard_authentication='
      },
    */
    if (httpAuth.isAuthenticated(request)) {
      return response.redirected({
        headers: {
          location: `${basePath}/app/kibana`
        }
      });
    }

    // Add the nextUrl to the redirect_uri as a parameter. The IDP uses the redirect_uri to redirect the user after successful authentication.
    // For example, it is used to redirect user to the correct dashboard if the user put shared URL in the browser address input before authentication.
    // To make this work, append the wildcard (*) to the valid redirect URI in the IDP configuration, for example
    // https://kibana.example.com:5601/auth/oidc/login*
    let nextUrl = null;
    try {
      if (request.url.searchParams.get('nextUrl') && decodeURIComponent(request.url.searchParams.get('nextUrl')) !== '/') {
        // Do not add the nextUrl to the redirect_uri to avoid the following breaking
        // change for the users that use normal URL to authenticate.
        nextUrl = (0, _sanitize_next_url.sanitizeNextUrl)(request.url.searchParams.get('nextUrl'));
      }
    } catch (error) {
      // We may have received a malformed URL, caught by decodedURIComponent.
      // In this case we just proceed without a nextUrl.
    }
    try {
      // We may have multiple SAML configurations.
      // The authTypeId may still be in the cookie. This happens when
      // a session token expires and no explicit logout is made. We need
      // this behaviour so that we can "refresh" the credentials from the IdP.
      const requestedAuthTypeId = request.url.searchParams.get('authTypeId') || sessionCookie.authTypeId;
      // Delete this again, otherwise the user won't get back to the login page
      // if trying to access Kibana again
      delete sessionCookie.authTypeId;
      delete sessionCookie.authType;
      const authConfigFinder = requestedAuthTypeId ? config => {
        return config.id === requestedAuthTypeId;
      } : config => {
        return config.method === 'saml';
      };
      const authConfig = (await searchGuardBackend.getAuthConfig(nextUrl)).auth_methods.find(authConfigFinder);
      if (!authConfig) {
        throw new Error('Auth config not found');
      }

      // When logging in, sessionCookie={}
      sessionCookie['temp-saml'] = {
        sso_context: authConfig.sso_context,
        authTypeId: authConfig.id || null
      };
      await sessionStorageFactory.asScoped(request).set(sessionCookie);
      return response.redirected({
        headers: {
          location: authConfig.sso_location
        }
      });
    } catch (error) {
      var _error$meta, _error$meta$body, _error$meta3, _error$meta3$body;
      logger.error(`SAML auth, fail to obtain the SAML header: ${error.stack}`);
      var headers = {
        location: basePath + '/login?err=saml_init'
      };
      var cookies = [];
      if ((_error$meta = error.meta) !== null && _error$meta !== void 0 && (_error$meta$body = _error$meta.body) !== null && _error$meta$body !== void 0 && _error$meta$body.error) {
        var _error$meta2, _error$meta2$body;
        cookies.push('sg_err=' + encodeURIComponent((_error$meta2 = error.meta) === null || _error$meta2 === void 0 ? void 0 : (_error$meta2$body = _error$meta2.body) === null || _error$meta2$body === void 0 ? void 0 : _error$meta2$body.error) + "; Path=/");
      } else if (error.message) {
        cookies.push('sg_err=' + encodeURIComponent(error.message) + "; Path=/");
      }
      if ((_error$meta3 = error.meta) !== null && _error$meta3 !== void 0 && (_error$meta3$body = _error$meta3.body) !== null && _error$meta3$body !== void 0 && _error$meta3$body.debug) {
        var _error$meta4, _error$meta4$body;
        cookies.push('sg_dbg=' + encodeURIComponent(JSON.stringify((_error$meta4 = error.meta) === null || _error$meta4 === void 0 ? void 0 : (_error$meta4$body = _error$meta4.body) === null || _error$meta4$body === void 0 ? void 0 : _error$meta4$body.debug)) + "; Path=/");
      }
      if (cookies.length > 0) {
        headers['set-cookie'] = cookies;
      }
      return response.redirected({
        headers
      });
    }
  });

  /*
  The page that the IDP redirects to after a successful SP-initiated login.
  */
  router.post({
    path: `${_constants.APP_ROOT}/searchguard/saml/acs`,
    options: {
      authRequired: false,
      xsrfRequired: false
    },
    validate: {
      body: _configSchema.schema.object({
        SAMLResponse: _configSchema.schema.string(),
        RelayState: _configSchema.schema.maybe(_configSchema.schema.string())
      }, {
        unknowns: 'allow'
      })
    }
  }, async (context, request, response) => {
    try {
      const {
        body: {
          SAMLResponse,
          RelayState
        } = {}
      } = request;
      /*
      When logging in,
      SAMLResponse = PHNhbWxwOlJlc3BvbnNlIHhtbG5zOnNhbWxwPSJ1cm46b2Fza...
      */

      const sessionCookie = await sessionStorageFactory.asScoped(request).get();
      if (!sessionCookie) {
        throw new Error('The session cookie is absent.');
      }
      /*
        When logging in,
        sessionCookie = {
          "temp-saml": {
            requestId: 'ONELOGIN_a0503578-1ae5-4621-80d5-49336f6d8673',
          }
        }
      */

      const {
        'temp-saml': storedRequestInfo,
        ...restSessionCookie
      } = sessionCookie;
      await sessionStorageFactory.asScoped(request).set(restSessionCookie);
      await authInstance.handleAuthenticate(request, {
        mode: 'saml',
        saml_response: SAMLResponse,
        sso_context: storedRequestInfo.sso_context,
        id: storedRequestInfo.authTypeId
      });
      const nextUrl = RelayState;
      if (nextUrl) {
        return response.redirected({
          headers: {
            location: (0, _sanitize_next_url.sanitizeNextUrl)(nextUrl, basePath)
          }
        });
      }
      return response.redirected({
        headers: {
          location: `${basePath}/app/home`
        }
      });
    } catch (error) {
      var _error$meta5, _error$meta5$body, _error$meta7, _error$meta7$body;
      logger.error(`SAML auth, failed to authorize: ${error.stack}`);
      var headers = {
        location: basePath + '/login?err=saml'
      };
      var cookies = [];
      if ((_error$meta5 = error.meta) !== null && _error$meta5 !== void 0 && (_error$meta5$body = _error$meta5.body) !== null && _error$meta5$body !== void 0 && _error$meta5$body.error) {
        var _error$meta6, _error$meta6$body;
        cookies.push('sg_err=' + encodeURIComponent((_error$meta6 = error.meta) === null || _error$meta6 === void 0 ? void 0 : (_error$meta6$body = _error$meta6.body) === null || _error$meta6$body === void 0 ? void 0 : _error$meta6$body.error) + "; Path=/");
      } else if (error.message) {
        cookies.push('sg_err=' + encodeURIComponent(error.message) + "; Path=/");
      }
      if ((_error$meta7 = error.meta) !== null && _error$meta7 !== void 0 && (_error$meta7$body = _error$meta7.body) !== null && _error$meta7$body !== void 0 && _error$meta7$body.debug) {
        var _error$meta8, _error$meta8$body;
        cookies.push('sg_dbg=' + encodeURIComponent(JSON.stringify((_error$meta8 = error.meta) === null || _error$meta8 === void 0 ? void 0 : (_error$meta8$body = _error$meta8.body) === null || _error$meta8$body === void 0 ? void 0 : _error$meta8$body.debug)) + "; Path=/");
      }
      if (cookies.length > 0) {
        headers['set-cookie'] = cookies;
      }
      return response.redirected({
        headers
      });
    }
  });

  /**
   * The page that the IdP redirects to after a successful IdP-initiated login
   */
  router.post({
    path: `${_constants.APP_ROOT}/searchguard/saml/acs/idpinitiated`,
    options: {
      authRequired: false,
      xsrfRequired: false
    },
    validate: {
      body: _configSchema.schema.object({
        SAMLResponse: _configSchema.schema.string()
      }, {
        unknowns: 'allow'
      })
    }
  }, async (context, request, response) => {
    try {
      await authInstance.handleAuthenticate(request, {
        mode: 'saml',
        saml_response: request.body.SAMLResponse,
        //sso_context: storedRequestInfo.sso_context,
        id: null
      });
      debugLog('Got SAMLResponse: ' + request.body.SAMLResponse);
      return response.redirected({
        headers: {
          location: `${basePath}/app/kibana`
        }
      });
    } catch (error) {
      var _error$meta9, _error$meta9$body, _error$meta11, _error$meta11$body;
      logger.error(`SAML IDP initiated authorization failed: ${error.stack}`);
      var headers = {
        location: basePath + '/login?err=saml_idpinitiated'
      };
      var cookies = [];
      if ((_error$meta9 = error.meta) !== null && _error$meta9 !== void 0 && (_error$meta9$body = _error$meta9.body) !== null && _error$meta9$body !== void 0 && _error$meta9$body.error) {
        var _error$meta10, _error$meta10$body;
        cookies.push('sg_err=' + encodeURIComponent((_error$meta10 = error.meta) === null || _error$meta10 === void 0 ? void 0 : (_error$meta10$body = _error$meta10.body) === null || _error$meta10$body === void 0 ? void 0 : _error$meta10$body.error) + "; Path=/");
      } else if (error.message) {
        cookies.push('sg_err=' + encodeURIComponent(error.message) + "; Path=/");
      }
      if ((_error$meta11 = error.meta) !== null && _error$meta11 !== void 0 && (_error$meta11$body = _error$meta11.body) !== null && _error$meta11$body !== void 0 && _error$meta11$body.debug) {
        var _error$meta12, _error$meta12$body;
        cookies.push('sg_dbg=' + encodeURIComponent(JSON.stringify((_error$meta12 = error.meta) === null || _error$meta12 === void 0 ? void 0 : (_error$meta12$body = _error$meta12.body) === null || _error$meta12$body === void 0 ? void 0 : _error$meta12$body.debug)) + "; Path=/");
      }
      if (cookies.length > 0) {
        headers['set-cookie'] = cookies;
      }
      return response.redirected({
        headers
      });
    }
  });

  /**
   * Redirect to logout page after an IdP redirect.
   * This is called after we log out from Kibana,
   * redirect to the IdP and then the IdP redirects
   * back to Kibana.
   */
  const logoutPath = `${_constants.APP_ROOT}/searchguard/saml/logout`;
  const logoutOptions = {
    authRequired: false,
    xsrfRequired: false
  };
  const logoutHandler = async (context, request, response) => {
    await authInstance.clear(request);
    return response.redirected({
      headers: {
        location: `${basePath}/`
      }
    });
  };
  // Logout route accepts both GET and POST
  router.get({
    path: logoutPath,
    options: logoutOptions,
    validate: false
  }, logoutHandler);
  router.post({
    path: logoutPath,
    options: logoutOptions,
    validate: false
  }, logoutHandler);
} //end module
//# sourceMappingURL=data:application/json;charset=utf-8;base64,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