"use strict";

Object.defineProperty(exports, "__esModule", {
  value: true
});
exports.ReadOnlyMode = void 0;
var _url = require("url");
var _lodash = require("lodash");
/* eslint-disable @kbn/eslint/require-license-header */

class ReadOnlyMode {
  constructor(logger) {
    this.logger = logger;
    this.readOnlyModeRoles = null;
    this.multiTenancyEnabled = false;
  }
  setupSync({
    kibanaCoreSetup,
    searchGuardBackend,
    configService
  }) {
    this.readOnlyModeRoles = configService.get('searchguard.readonly_mode.roles');
    this.multiTenancyEnabled = configService.get('searchguard.multitenancy.enabled');
    if (this.readOnlyModeRoles.length || this.multiTenancyEnabled) {
      this.registerSwitcher({
        kibanaCoreSetup,
        searchGuardBackend,
        configService
      });
    }
  }
  hasMultipleTenants(tenantsObject, globalTenantEnabled, privateTenantEnabled, userName) {
    const tenantsCopy = {
      ...tenantsObject
    };
    if (!globalTenantEnabled) {
      delete tenantsCopy.SGS_GLOBAL_TENANT;
    }
    if (!privateTenantEnabled) {
      delete tenantsCopy[userName];
    }
    return Object.keys(tenantsCopy).length > 1 ? true : false;
  }
  isAnonymousPage(request) {
    if (request.headers && request.headers.referer) {
      try {
        const {
          pathname
        } = (0, _url.parse)(request.headers.referer);
        const pathsToIgnore = ['login', 'logout', 'customerror'];
        if (pathsToIgnore.indexOf(pathname.split('/').pop()) > -1) {
          return true;
        }
      } catch (error) {
        this.logger.error(`Could not parse the referer for the capabilites: ${error.stack}`);
      }
    }
    return false;
  }
  switcherHandler({
    searchGuardBackend,
    configService
  }) {
    return async (request, uiCapabilities) => {
      // Ignore for non authenticated paths
      if (this.isAnonymousPage(request)) {
        return uiCapabilities;
      }
      try {
        const authInfo = await searchGuardBackend.authinfo(request.headers);
        if (this.hasReadOnlyRole(authInfo, this.readOnlyModeRoles)) {
          // A read only role trumps the tenant access rights
          return this.toggleForReadOnlyRole(uiCapabilities, configService, authInfo);
        } else if (this.isReadOnlyTenant(authInfo)) {
          return this.toggleForReadOnlyTenant(uiCapabilities, configService);
        }
      } catch (error) {
        this.logger.error(`Could not check auth info: ${error.stack}`);
      }
      return uiCapabilities;
    };
  }
  registerSwitcher({
    kibanaCoreSetup,
    searchGuardBackend,
    configService
  }) {
    kibanaCoreSetup.capabilities.registerSwitcher(this.switcherHandler({
      searchGuardBackend,
      configService
    }));
  }
  hasReadOnlyRole(authInfo, readOnlyModeRoles) {
    return authInfo.sg_roles.some(role => readOnlyModeRoles.includes(role));
  }

  /**
   * Check if current tenant is read only for the user
   * @param authInfo
   * @returns {boolean}
   */
  isReadOnlyTenant(authInfo) {
    // The global tenant would be '' == falsey
    const currentTenant = authInfo.user_requested_tenant || 'SGS_GLOBAL_TENANT';
    if (currentTenant === '__user__') {
      // We don't limit the private tenant
      return false;
    }
    const isReadOnlyTenant = authInfo.sg_tenants[currentTenant] !== true ? true : false;
    return isReadOnlyTenant;
  }
  toggleForReadOnlyRole(uiCapabilities, config, authInfo) {
    const globalTenantEnabled = config.get('searchguard.multitenancy.tenants.enable_global');
    const privateTenantEnabled = config.get('searchguard.multitenancy.tenants.enable_private');
    const whitelist = ['home', 'dashboard', 'dashboards'];

    // Show the MT app if user has more than one tenant
    if (this.hasMultipleTenants(authInfo.sg_tenants, globalTenantEnabled, privateTenantEnabled, authInfo.user_name)) {
      whitelist.push('searchguard-multitenancy');
    }
    Object.keys(uiCapabilities).forEach(capability => {
      if (capability === 'navLinks') {
        // Hide navLinks
        Object.keys(uiCapabilities.navLinks).forEach(navLinkId => {
          uiCapabilities.navLinks[navLinkId] = whitelist.indexOf(navLinkId) > -1;
        });
      } else if (capability === 'catalogue') {
        // Hide features from the catalogue
        Object.keys(uiCapabilities.catalogue).forEach(appId => {
          uiCapabilities.catalogue[appId] = whitelist.indexOf(appId) > -1;
        });
      } else if (
      // Here we're looking for a show property
      typeof uiCapabilities[capability] === 'object' && typeof uiCapabilities[capability].show !== 'undefined') {
        // If an app has show = false, Kibana will redirect away from its url.
        uiCapabilities[capability].show = whitelist.indexOf(capability) > -1;
      }
    });
    const defaultUICapabilities = {
      dashboard: {
        createNew: false,
        showWriteControls: false,
        saveQuery: false
      }
    };
    const finalCapabilities = (0, _lodash.merge)(uiCapabilities, defaultUICapabilities);
    return finalCapabilities;
  }
  toggleForReadOnlyTenant(uiCapabilities) {
    const defaultTenantOnlyCapabilities = {
      navLinks: {
        'kibana:stack_management': false,
        management: false
      },
      catalogue: {
        advanced_settings: false,
        index_patterns: false
      },
      dashboard: {
        createNew: false,
        showWriteControls: false,
        saveQuery: false
      },
      visualize: {
        createShortUrl: false,
        delete: false,
        save: false,
        saveQuery: false
      },
      management: {
        kibana: {
          indexPatterns: false
        }
      }
    };
    const finalCapabilities = (0, _lodash.merge)(uiCapabilities, defaultTenantOnlyCapabilities);
    return finalCapabilities;
  }
}
exports.ReadOnlyMode = ReadOnlyMode;
//# sourceMappingURL=data:application/json;charset=utf-8;base64,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