"use strict";

var _interopRequireDefault = require("@babel/runtime/helpers/interopRequireDefault");
Object.defineProperty(exports, "__esModule", {
  value: true
});
exports.default = void 0;
var _defineProperty2 = _interopRequireDefault(require("@babel/runtime/helpers/defineProperty"));
var _lodash = _interopRequireDefault(require("lodash"));
var _authentication_error = _interopRequireDefault(require("../auth/errors/authentication_error"));
var _user = _interopRequireDefault(require("../auth/user"));
/*
 *    Copyright 2021 floragunn GmbH
 *
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 * http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 */

/**
 * The SearchGuard  backend.
 */
class SearchGuardBackend {
  constructor({
    elasticsearch,
    configService,
    core
  }) {
    (0, _defineProperty2.default)(this, "_client", async ({
      headers = {},
      asWho = 'asCurrentUser',
      ...options
    }) => {
      const {
        body
      } = await this.elasticsearch.client.asScoped({
        headers
      })[asWho].transport.request(options);
      return body;
    });
    (0, _defineProperty2.default)(this, "getAuthConfig", async (nextUrl = null) => {
      try {
        const sgFrontendConfigId = this.configService.get('searchguard.sg_frontend_config_id') || 'default';
        let frontendBaseUrl = this.configService.get('searchguard.frontend_base_url') || this.core.http.basePath.publicBaseUrl;
        if (!frontendBaseUrl) {
          let serverInfo = this.core.http.getServerInfo();
          frontendBaseUrl = serverInfo.protocol + "://" + serverInfo.hostname + ":" + serverInfo.port + "/" + this.core.http.basePath.serverBasePath;
        }
        const response = await this._client({
          path: '/_searchguard/auth/config',
          method: 'POST',
          asWho: 'asInternalUser',
          body: {
            config_id: sgFrontendConfigId,
            frontend_base_url: frontendBaseUrl,
            next_url: nextUrl
          }
        });
        return response;
      } catch (error) {
        if (error.statusCode === 401) {
          throw new _authentication_error.default('Invalid username or password', error);
        }
        throw error;
      }
    });
    (0, _defineProperty2.default)(this, "authenticate", async credentials => {
      const authHeader = Buffer.from(`${credentials.username}:${credentials.password}`).toString('base64');
      try {
        const response = await this._client({
          path: '/_searchguard/authinfo',
          method: 'get',
          headers: {
            authorization: `Basic ${authHeader}`
          }
        });
        return new _user.default(credentials.username, credentials, credentials, response.sg_roles, response.backend_roles, response.sg_tenants, response.user_requested_tenant);
      } catch (error) {
        if (error.statusCode === 401) {
          throw new _authentication_error.default('Invalid username or password', error);
        }
        throw error;
      }
    });
    (0, _defineProperty2.default)(this, "authenticateWithHeader", async (headerName, headerValue, additionalAuthHeaders = {}) => {
      try {
        const credentials = {
          headerName: headerName,
          headerValue: headerValue
        };
        const headers = {
          ...additionalAuthHeaders
        };

        // For anonymous auth, we wouldn't have any value here
        if (headerValue) {
          headers[headerName] = headerValue;
        }
        const response = await this._client({
          path: '/_searchguard/authinfo',
          method: 'get',
          headers
        });
        return new _user.default(response.user_name, credentials, null, response.sg_roles, response.backend_roles, response.sg_tenants, response.user_requested_tenant);
      } catch (error) {
        if (error.statusCode === 401) {
          throw new _authentication_error.default('Invalid username or password', error);
        }
        throw error;
      }
    });
    /**
     * A wrapper for authinfo() when we expect a response to be used for a cookie
     * @param headers
     * @param credentials
     * @returns {Promise<User>}
     */
    (0, _defineProperty2.default)(this, "authenticateWithHeaders", async (headers, credentials = {}, additionalAuthHeaders = {}) => {
      headers = {
        ...additionalAuthHeaders,
        ...headers
      };
      try {
        const response = await this._client({
          path: '/_searchguard/authinfo',
          method: 'get',
          headers
        });
        return new _user.default(response.user_name, credentials, null, response.sg_roles, response.backend_roles, response.sg_tenants, response.user_requested_tenant);
      } catch (error) {
        if (error.statusCode === 401) {
          throw new _authentication_error.default('Invalid username or password', error);
        }
        throw error;
      }
    });
    (0, _defineProperty2.default)(this, "createSessionWithHeaders", async (headers, additionalAuthHeaders = {}) => {
      headers = {
        ...additionalAuthHeaders,
        ...headers
      };
      try {
        return await this._client({
          path: '/_searchguard/auth/session/with_header',
          method: 'POST',
          headers
        });
      } catch (error) {
        console.log(error);
        throw error;
      }
    });
    (0, _defineProperty2.default)(this, "authinfo", async headers => {
      try {
        return await this._client({
          path: '/_searchguard/authinfo',
          method: 'get',
          headers
        });
      } catch (error) {
        if (error.statusCode === 401) {
          throw new _authentication_error.default(error.message, error);
        }
        throw error;
      }
    });
    (0, _defineProperty2.default)(this, "sessionInfo", async headers => {
      try {
        return await this._client({
          path: '/_searchguard/auth/session',
          method: 'get',
          headers
        });
      } catch (error) {
        if (error.statusCode === 401) {
          throw new _authentication_error.default(error.message, error);
        }
        throw error;
      }
    });
    (0, _defineProperty2.default)(this, "getKibanaInfoWithInternalUser", async () => {
      try {
        return await this._client({
          path: '/_searchguard/kibanainfo',
          method: 'get',
          asWho: 'asInternalUser'
        });
      } catch (error) {
        if (error.statusCode === 401) {
          throw new _authentication_error.default(error.message, error);
        }
        throw error;
      }
    });
    /**
     * Check for application permissions
     * @param headers
     * @param permissions
     * @returns {Promise<*>}
     */
    (0, _defineProperty2.default)(this, "hasPermissions", async (headers, permissions) => {
      try {
        return await this._client({
          path: '/_searchguard/permission',
          method: 'get',
          headers,
          querystring: {
            permissions
          }
        });
      } catch (error) {
        if (error.statusCode === 401) {
          throw new _authentication_error.default(error.message, error);
        }
        throw error;
      }
    });
    (0, _defineProperty2.default)(this, "multitenancyinfo", async headers => {
      try {
        return await this._client({
          path: '/_searchguard/kibanainfo',
          method: 'get',
          headers
        });
      } catch (error) {
        if (error.statusCode === 401) {
          throw new _authentication_error.default(error.message, error);
        }
        throw error;
      }
    });
    (0, _defineProperty2.default)(this, "systeminfo", async headers => {
      try {
        return await this._client({
          path: '/_searchguard/license',
          method: 'get',
          headers
        });
      } catch (error) {
        if (error.statusCode === 401) {
          throw new _authentication_error.default(error.message, error);
        }
        throw error;
      }
    });
    (0, _defineProperty2.default)(this, "getTenantInfoWithInternalUser", async () => {
      try {
        return await this._client({
          path: '/_searchguard/tenantinfo',
          method: 'get',
          asWho: 'asInternalUser'
        });
      } catch (error) {
        if (error.statusCode === 401) {
          throw new _authentication_error.default(error.message, error);
        }
        throw error;
      }
    });
    (0, _defineProperty2.default)(this, "getTenantInfo", async headers => {
      try {
        return await this._client({
          path: '/_searchguard/tenantinfo',
          method: 'get',
          headers
        });
      } catch (error) {
        if (error.statusCode === 401) {
          throw new _authentication_error.default(error.message, error);
        }
        throw error;
      }
    });
    (0, _defineProperty2.default)(this, "uploadLicense", async (headers, body) => {
      try {
        return await this._client({
          path: '/_searchguard/api/license',
          method: 'put',
          headers,
          body
        });
      } catch (error) {
        if (error.statusCode === 401) {
          throw new _authentication_error.default(error.message, error);
        }
        throw error;
      }
    });
    /**
     * @deprecated, use the sessionPlugin instead
     * @param user
     * @returns {Promise<{authorization: string}>}
     */
    (0, _defineProperty2.default)(this, "getAuthHeaders", async user => {
      const credentials = user.credentials;
      const authHeader = Buffer.from(`${credentials.username}:${credentials.password}`).toString('base64');
      return {
        authorization: `Basic ${authHeader}`
      };
    });
    this.elasticsearch = elasticsearch;
    this.configService = configService;
    this.core = core;
  }
  async authenticateWithSession(credentials) {
    try {
      const response = await this._client({
        path: '/_searchguard/auth/session',
        method: 'POST',
        body: credentials
      });
      return response;
    } catch (error) {
      // TODO remove
      console.log(error);
      throw error;
    }
  }
  async logoutSession(headers) {
    try {
      return await this._client({
        path: '/_searchguard/auth/session',
        method: 'DELETE',
        headers
      });
    } catch (error) {
      if (error.statusCode === 401) {
        throw new _authentication_error.default('Invalid username or password', error);
      }
      throw error;
    }
  }
  buildSessionResponse(credentials, authInfoResponse) {
    return new _user.default(authInfoResponse.user_name, credentials, null, authInfoResponse.sg_roles, authInfoResponse.backend_roles, authInfoResponse.sg_tenants, authInfoResponse.user_requested_tenant);
  }
  getAuthHeaders(username, password) {
    const authHeader = Buffer.from(`${username}:${password}`).toString('base64');
    return {
      authorization: `Basic ${authHeader}`
    };
  }
  getUser(username, password) {
    const credentials = {
      username: username,
      password: password
    };
    const user = new _user.default(credentials.username, credentials, credentials, [], {});
    return user;
  }
  updateAndGetTenantPreferences(request, user, tenant) {
    /*
    const preferencesCookieName = this.configService.get(
      'searchguard.cookie.preferences_cookie_name'
    );
      */

    //const prefs = request.state[preferencesCookieName];
    const prefs = {};
    // no prefs cookie present
    if (!prefs) {
      const newPrefs = {};
      newPrefs[user] = tenant;
      return newPrefs;
    }
    prefs[user] = tenant;
    return prefs;
  }
  getTenantByPreference(request, username, tenants, preferredTenants, globalEnabled, privateEnabled) {
    // delete user from tenants first to check if we have a tenant to choose from at all
    // keep original preferences untouched, we need the original values again
    // http://stackoverflow.com/questions/728360/how-do-i-correctly-clone-a-javascript-object
    const tenantsCopy = JSON.parse(JSON.stringify(tenants));
    delete tenantsCopy[username];

    // sanity check
    if (!globalEnabled && !privateEnabled && _lodash.default.isEmpty(tenantsCopy)) {
      return null;
    }

    // Evaluate preferredTenants from kibana config
    if (preferredTenants && !_lodash.default.isEmpty(preferredTenants)) {
      for (let i = 0; i < preferredTenants.length; i++) {
        const check = preferredTenants[i].toLowerCase();
        if (globalEnabled && (check === 'global' || check === '__global__')) {
          return '';
        }
        if (privateEnabled && (check === 'private' || check === '__user__') && tenants[username] !== undefined) {
          return '__user__';
        }
        if (tenants[check] !== undefined) {
          return check;
        }
        if (check.toLowerCase() === 'private' && privateEnabled) {
          return '__user__';
        }
      }
    }

    // no pref in cookie, no preferred tenant in kibana, use GLOBAL, Private or the first tenant in the list
    if (globalEnabled) {
      return '';
    }
    if (privateEnabled) {
      return '__user__';
    } else {
      delete tenants[username];
    }

    // sort tenants by putting the keys in an array first
    let tenantkeys = [];
    let k;
    for (k in tenants) {
      if (tenants.hasOwnProperty(k)) {
        tenantkeys.push(k);
      }
    }
    tenantkeys.sort();
    if (!globalEnabled) {
      tenantkeys = tenantkeys.filter(tenantKey => tenantKey !== 'SGS_GLOBAL_TENANT');
    }
    return tenantkeys[0];
  }
  validateTenant(username, requestedTenant, tenants, globalEnabled, privateEnabled) {
    // delete user from tenants first to check if we have a tenant to choose from at all
    // keep original preferences untouched, we need the original values again
    // http://stackoverflow.com/questions/728360/how-do-i-correctly-clone-a-javascript-object
    const tenantsCopy = JSON.parse(JSON.stringify(tenants));
    delete tenantsCopy[username];
    if (!globalEnabled) {
      delete tenantsCopy.SGS_GLOBAL_TENANT;
    }

    // sanity check: no global, no private, no other tenants -> no tenant available
    if (!globalEnabled && !privateEnabled && _lodash.default.isEmpty(tenantsCopy)) {
      return null;
    }

    // requested tenant accessible for user
    if (tenants[requestedTenant] !== undefined) {
      return requestedTenant;
    }
    if ((requestedTenant === '__user__' || requestedTenant === 'private') && tenants[username] && privateEnabled) {
      return '__user__';
    }
    if ((requestedTenant === 'global' || requestedTenant === '') && globalEnabled) {
      return '';
    }
    return null;
  }
}
exports.default = SearchGuardBackend;
module.exports = exports.default;
//# sourceMappingURL=data:application/json;charset=utf-8;base64,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