"use strict";

var _interopRequireDefault = require("@babel/runtime/helpers/interopRequireDefault");
Object.defineProperty(exports, "__esModule", {
  value: true
});
exports.default = void 0;
var _defineProperty2 = _interopRequireDefault(require("@babel/runtime/helpers/defineProperty"));
var _errors = require("../../auth/errors");
var _not_found = _interopRequireDefault(require("./../../backend/errors/not_found"));
/*
 *    Copyright 2020 floragunn GmbH
 *
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 * http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 */

/**
 * The SearchGuard  backend.
 */
class SearchGuardConfigurationBackend {
  constructor({
    elasticsearch
  }) {
    (0, _defineProperty2.default)(this, "_client", async ({
      headers = {},
      asWho = 'asCurrentUser',
      ...options
    }) => {
      const {
        body
      } = await this.elasticsearch.client.asScoped({
        headers
      })[asWho].transport.request(options);
      return body;
    });
    (0, _defineProperty2.default)(this, "restapiinfo", async headers => {
      try {
        return await this._client({
          path: '/_searchguard/api/permissionsinfo',
          method: 'get',
          headers: headers
        });
      } catch (error) {
        if (error.statusCode === 401) {
          throw new _errors.AuthenticationError(error.message, error);
        }
        throw error;
      }
    });
    (0, _defineProperty2.default)(this, "indices", async ({
      headers,
      index = []
    } = {}) => {
      try {
        const {
          body: response
        } = await this.elasticsearch.client.asScoped({
          headers
        }).asCurrentUser.cat.indices({
          index,
          format: 'json',
          h: 'index,health'
        });
        return response;
      } catch (error) {
        if (error.statusCode === 401) {
          throw new _errors.AuthenticationError(error.message, error);
        }
        throw error;
      }
    });
    (0, _defineProperty2.default)(this, "aliases", async ({
      headers,
      alias = []
    } = {}) => {
      try {
        const {
          body: response
        } = await this.elasticsearch.client.asScoped({
          headers
        }).asCurrentUser.cat.aliases({
          alias,
          format: 'json',
          h: 'index,alias'
        });
        return response;
      } catch (error) {
        if (error.statusCode === 401) {
          throw new _errors.AuthenticationError(error.message, error);
        }
        throw error;
      }
    });
    (0, _defineProperty2.default)(this, "list", async (headers, resourceName) => {
      try {
        return await this._client({
          path: `/_searchguard/api/${resourceName}`,
          method: 'get',
          headers
        });
      } catch (error) {
        if (error.statusCode === 401) {
          throw new _errors.AuthenticationError(error.message, error);
        }
        throw error;
      }
    });
    (0, _defineProperty2.default)(this, "get", async (headers, resourceName, id) => {
      try {
        const response = await this._client({
          path: `/_searchguard/api/${resourceName}/${encodeURIComponent(id)}`,
          method: 'get',
          headers
        });
        return response[id];
      } catch (error) {
        if (error.statusCode === 401) {
          throw new _errors.AuthenticationError(error.message, error);
        }
        if (error.statusCode === 404) {
          throw new _not_found.default(error.message, error);
        }
        throw error;
      }
    });
    (0, _defineProperty2.default)(this, "save", async (headers, resourceName, id, body) => {
      try {
        return await this._client({
          path: `/_searchguard/api/${resourceName}/${encodeURIComponent(id)}`,
          method: 'put',
          headers,
          body
        });
      } catch (error) {
        if (error.statusCode === 401) {
          throw new _errors.AuthenticationError(error.message, error);
        }
        if (error.statusCode === 404) {
          throw new _not_found.default(error.message, error);
        }
        throw error;
      }
    });
    (0, _defineProperty2.default)(this, "delete", async (headers, resourceName, id) => {
      try {
        return await this._client({
          path: `/_searchguard/api/${resourceName}/${encodeURIComponent(id)}`,
          method: 'delete',
          headers
        });
      } catch (error) {
        if (error.statusCode === 401) {
          throw new _errors.AuthenticationError(error.message, error);
        }
        if (error.statusCode === 404) {
          throw new _not_found.default(error.message, error);
        }
        throw error;
      }
    });
    (0, _defineProperty2.default)(this, "clearCache", async headers => {
      try {
        return await this._client({
          path: '/_searchguard/api/cache',
          method: 'delete',
          headers
        });
      } catch (error) {
        if (error.statusCode === 401) {
          throw new _errors.AuthenticationError(error.message, error);
        }
        throw error;
      }
    });
    (0, _defineProperty2.default)(this, "validateDls", async (headers, indexname, body) => {
      try {
        return await this._client({
          path: '/_validate/query?explain=true',
          method: 'post',
          headers,
          body
        });
      } catch (error) {
        if (error.statusCode === 401) {
          throw new _errors.AuthenticationError(error.message, error);
        }
        throw error;
      }
    });
    (0, _defineProperty2.default)(this, "getIndexMappings", async ({
      headers,
      body: {
        index
      }
    }) => {
      try {
        const {
          body: mappings
        } = await this.elasticsearch.client.asScoped({
          headers
        }).asCurrentUser.indices.getMapping({
          index: index.join(','),
          ignore_unavailable: true,
          allow_no_indices: true
        });
        return {
          total: Object.keys(mappings).length,
          mappings
        };
      } catch (error) {
        if (error.statusCode === 401) {
          throw new _errors.AuthenticationError(error.message, error);
        }
        throw error;
      }
    });
    this.elasticsearch = elasticsearch;
  }
}
exports.default = SearchGuardConfigurationBackend;
module.exports = exports.default;
//# sourceMappingURL=data:application/json;charset=utf-8;base64,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