"use strict";

Object.defineProperty(exports, "__esModule", {
  value: true
});
exports.SearchGuard = void 0;
var _routes = require("./routes");
var _auth = require("./auth");
var _routes2 = require("./system/routes");
var _routes3 = require("./configuration/routes/routes");
var _sanity_checks = require("./sanity_checks");
var _security_cookie = require("./session/security_cookie");
var _ReadOnlyMode = require("./authorization/ReadOnlyMode");
var _AuthManager = require("./auth/AuthManager");
var _routes_auth = require("./auth/routes_auth");
/* eslint-disable @kbn/eslint/require-license-header */

class SearchGuard {
  constructor(coreContext) {
    this.coreContext = coreContext;
    this.logger = this.coreContext.logger.get('searchguard');
  }
  async setup({
    core,
    pluginDependencies,
    configService,
    kibanaRouter,
    searchGuardBackend,
    searchGuardConfigurationBackend,
    spacesService,
    elasticsearch
  }) {
    this.logger.debug('Setup app');
    const kibanaVersionIndex = configService.get('kibana.index') + '_' + this.coreContext.env.packageInfo.version;
    elasticsearch.client.rootScopedClient.on('request', (0, _auth.rootScopedClientRequestWrapper)({
      configService,
      kibanaVersionIndex
    }));
    try {
      // Sanity checks
      (0, _sanity_checks.checkTLSConfig)({
        configService,
        logger: this.logger
      });
      (0, _sanity_checks.checkCookieConfig)({
        configService,
        logger: this.logger
      });
      (0, _sanity_checks.checkXPackSecurityDisabled)({
        pluginDependencies,
        logger: this.logger
      });
      (0, _sanity_checks.checkDoNotFailOnForbidden)({
        searchGuardBackend,
        logger: this.logger
      });

      // Inits the authInfo route
      (0, _auth.defineAuthInfoRoutes)({
        searchGuardBackend,
        kibanaCore: core,
        logger: this.logger
      });
      (0, _routes.registerRoutes)({
        router: kibanaRouter,
        config: configService.getConfig(),
        logger: this.logger
      });
      const cookieOptions = (0, _security_cookie.getSecurityCookieOptions)(configService);
      const sessionStorageFactory = await core.http.createCookieSessionStorageFactory(cookieOptions);

      // We must extend the cookie options.
      // Because Kibana doesn't support all the options we need.
      (0, _security_cookie.extendSecurityCookieOptions)(cookieOptions);
      const authType = configService.get('searchguard.auth.type', null);
      let authManager = null;
      let kerberos = null;

      // Handle Kerberos separately because we don't want to bring up entire jungle from AuthType here.
      if (authType === 'kerberos') {
        kerberos = new _auth.Kerberos({
          pluginDependencies,
          config: configService,
          searchGuardBackend,
          logger: this.coreContext.logger.get('searchguard-kerberos-auth'),
          basePath: core.http.basePath.get(),
          sessionStorageFactory: sessionStorageFactory
        });
        core.http.registerAuth(kerberos.checkAuth);
      } else if (authType !== 'proxy') {
        authManager = new _AuthManager.AuthManager({
          kibanaCore: core,
          sessionStorageFactory,
          pluginDependencies,
          logger: this.coreContext.logger.get('searchguard-auth'),
          searchGuardBackend,
          configService,
          spacesService
        });
        authManager.registerAuthInstances();
        (0, _routes_auth.defineAuthRoutes)({
          kibanaCore: core,
          authManager,
          searchGuardBackend,
          configService
        });
        if (authManager) {
          // authManager.onPreAuth needs to run before any other handler
          // that manipulates the request headers (e.g. MT)
          core.http.registerOnPreAuth(authManager.onPreAuth);
          core.http.registerAuth(authManager.checkAuth);
          core.http.registerOnPostAuth(authManager.onPostAuth);
        }
      }
      if (authType && ['proxy', 'kerberos'].indexOf(authType) === -1) {
        try {
          this.logger.info('Initialising Search Guard authentication plugin.');
          if (configService.get('searchguard.cookie.password') === 'searchguard_cookie_default_password') {
            this.logger.warn("Default cookie password detected, please set a password in kibana.yml by setting 'searchguard.cookie.password' (min. 32 characters).");
          }
          if (!configService.get('searchguard.cookie.secure')) {
            this.logger.warn("'searchguard.cookie.secure' is set to false, cookies are transmitted over unsecure HTTP connection. Consider using HTTPS and set this key to 'true'");
          }
        } catch (error) {
          this.logger.error(`An error occurred registering server plugins: ${error}`);
          throw error;
        }
      }

      // @todo TEST
      if (configService.get('searchguard.xff.enabled')) {
        require('./xff/xff')(core);
        this.logger.info('Search Guard XFF enabled.');
      }
      if (configService.get('searchguard.configuration.enabled')) {
        (0, _routes3.defineConfigurationRoutes)({
          searchGuardConfigurationBackend,
          logger: this.logger,
          kibanaCore: core
        });
        this.logger.info('Routes for Search Guard configuration GUI registered. This is an Enterprise feature.');
      } else {
        this.logger.warn('Search Guard configuration GUI disabled');
      }
      (0, _routes2.defineSystemRoutes)({
        searchGuardBackend,
        logger: this.logger,
        kibanaCore: core
      });
      this.logger.info('Search Guard system routes registered.');
      if (configService.get('searchguard.readonly_mode.enabled')) {
        const readOnlyMode = new _ReadOnlyMode.ReadOnlyMode(this.coreContext.logger.get('searchguard-readonly'));
        readOnlyMode.setupSync({
          kibanaCoreSetup: core,
          searchGuardBackend,
          configService
        });
      }
      return {
        authManager,
        sessionStorageFactory,
        kerberos
      };
    } catch (error) {
      this.logger.error(error);
      throw error;
    }
  }
}
exports.SearchGuard = SearchGuard;
//# sourceMappingURL=data:application/json;charset=utf-8;base64,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