"use strict";

Object.defineProperty(exports, "__esModule", {
  value: true
});
exports.getAlertsRoute = exports.getAlerts = void 0;
var _configSchema = require("@kbn/config-schema");
var _lib = require("../../lib");
var _constants = require("../../../../../common/signals/constants");
/*
 *    Copyright 2020 floragunn GmbH
 *
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 * http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 */

const getAlerts = ({
  clusterClient,
  fetchAllFromScroll,
  logger
}) => async (context, request, response) => {
  try {
    const {
      body: {
        query,
        sort,
        index,
        scroll
      },
      headers: {
        sgtenant
      }
    } = request;
    const options = {
      index,
      scroll
    };
    if (query && !!Object.keys(query).length) {
      // We don't filter alerts by tenant if it is Global tenant (value is '')
      if (sgtenant) {
        if (!query.bool.must) {
          query.bool.must = [];
        }
        query.bool.must.push({
          term: {
            'tenant.keyword': {
              value: sgtenant
            }
          }
        });
      }
      options.body = {
        sort,
        query
      };
    }
    const {
      body: firstScrollResponse
    } = await clusterClient.asScoped(request).asCurrentUser.search(options);
    const hits = await fetchAllFromScroll({
      clusterClient,
      scroll,
      request,
      response: firstScrollResponse
    });
    return response.ok({
      body: {
        ok: true,
        resp: hits.map(h => ({
          ...h._source,
          _id: h._id,
          _index: h._index
        }))
      }
    });
  } catch (err) {
    logger.error(`getAlerts: ${err.stack}`);
    return response.customError((0, _lib.serverError)(err));
  }
};
exports.getAlerts = getAlerts;
const getAlertsRoute = ({
  router,
  clusterClient,
  fetchAllFromScroll,
  logger
}) => {
  router.post({
    path: _constants.ROUTE_PATH.ALERTS,
    validate: {
      headers: _configSchema.schema.object({
        sgtenant: _configSchema.schema.string({
          defaultValue: _constants.NO_MULTITENANCY_TENANT
        })
      }, {
        unknowns: 'allow'
      }),
      body: _configSchema.schema.object({
        index: _configSchema.schema.string({
          defaultValue: _constants.INDEX.ALERTS
        }),
        scroll: _configSchema.schema.string({
          defaultValue: _constants.ES_SCROLL_SETTINGS.KEEPALIVE
        }),
        query: _configSchema.schema.object({}, {
          unknowns: 'allow'
        }),
        sort: _configSchema.schema.arrayOf(_configSchema.schema.object({
          [_constants.DEFAULT_DATEFIELD]: _configSchema.schema.string({
            validate(value) {
              const accepted = ['desc', 'asc'];
              if (!accepted.includes(value)) {
                return `The accepted values: ${accepted.join(', ')}`;
              }
            }
          })
        }, {
          unknowns: 'allow'
        }), {
          defaultValue: {
            [_constants.DEFAULT_DATEFIELD]: 'desc'
          }
        })
      })
    }
  }, getAlerts({
    clusterClient,
    fetchAllFromScroll,
    logger
  }));
};
exports.getAlertsRoute = getAlertsRoute;
//# sourceMappingURL=data:application/json;charset=utf-8;base64,eyJ2ZXJzaW9uIjozLCJuYW1lcyI6WyJfY29uZmlnU2NoZW1hIiwicmVxdWlyZSIsIl9saWIiLCJfY29uc3RhbnRzIiwiZ2V0QWxlcnRzIiwiY2x1c3RlckNsaWVudCIsImZldGNoQWxsRnJvbVNjcm9sbCIsImxvZ2dlciIsImNvbnRleHQiLCJyZXF1ZXN0IiwicmVzcG9uc2UiLCJib2R5IiwicXVlcnkiLCJzb3J0IiwiaW5kZXgiLCJzY3JvbGwiLCJoZWFkZXJzIiwic2d0ZW5hbnQiLCJvcHRpb25zIiwiT2JqZWN0Iiwia2V5cyIsImxlbmd0aCIsImJvb2wiLCJtdXN0IiwicHVzaCIsInRlcm0iLCJ2YWx1ZSIsImZpcnN0U2Nyb2xsUmVzcG9uc2UiLCJhc1Njb3BlZCIsImFzQ3VycmVudFVzZXIiLCJzZWFyY2giLCJoaXRzIiwib2siLCJyZXNwIiwibWFwIiwiaCIsIl9zb3VyY2UiLCJfaWQiLCJfaW5kZXgiLCJlcnIiLCJlcnJvciIsInN0YWNrIiwiY3VzdG9tRXJyb3IiLCJzZXJ2ZXJFcnJvciIsImV4cG9ydHMiLCJnZXRBbGVydHNSb3V0ZSIsInJvdXRlciIsInBvc3QiLCJwYXRoIiwiUk9VVEVfUEFUSCIsIkFMRVJUUyIsInZhbGlkYXRlIiwic2NoZW1hIiwib2JqZWN0Iiwic3RyaW5nIiwiZGVmYXVsdFZhbHVlIiwiTk9fTVVMVElURU5BTkNZX1RFTkFOVCIsInVua25vd25zIiwiSU5ERVgiLCJFU19TQ1JPTExfU0VUVElOR1MiLCJLRUVQQUxJVkUiLCJhcnJheU9mIiwiREVGQVVMVF9EQVRFRklFTEQiLCJhY2NlcHRlZCIsImluY2x1ZGVzIiwiam9pbiJdLCJzb3VyY2VzIjpbImdldC5qcyJdLCJzb3VyY2VzQ29udGVudCI6WyIvKlxuICogICAgQ29weXJpZ2h0IDIwMjAgZmxvcmFndW5uIEdtYkhcbiAqXG4gKiBMaWNlbnNlZCB1bmRlciB0aGUgQXBhY2hlIExpY2Vuc2UsIFZlcnNpb24gMi4wICh0aGUgXCJMaWNlbnNlXCIpO1xuICogeW91IG1heSBub3QgdXNlIHRoaXMgZmlsZSBleGNlcHQgaW4gY29tcGxpYW5jZSB3aXRoIHRoZSBMaWNlbnNlLlxuICogWW91IG1heSBvYnRhaW4gYSBjb3B5IG9mIHRoZSBMaWNlbnNlIGF0XG4gKlxuICogaHR0cDovL3d3dy5hcGFjaGUub3JnL2xpY2Vuc2VzL0xJQ0VOU0UtMi4wXG4gKlxuICogVW5sZXNzIHJlcXVpcmVkIGJ5IGFwcGxpY2FibGUgbGF3IG9yIGFncmVlZCB0byBpbiB3cml0aW5nLCBzb2Z0d2FyZVxuICogZGlzdHJpYnV0ZWQgdW5kZXIgdGhlIExpY2Vuc2UgaXMgZGlzdHJpYnV0ZWQgb24gYW4gXCJBUyBJU1wiIEJBU0lTLFxuICogV0lUSE9VVCBXQVJSQU5USUVTIE9SIENPTkRJVElPTlMgT0YgQU5ZIEtJTkQsIGVpdGhlciBleHByZXNzIG9yIGltcGxpZWQuXG4gKiBTZWUgdGhlIExpY2Vuc2UgZm9yIHRoZSBzcGVjaWZpYyBsYW5ndWFnZSBnb3Zlcm5pbmcgcGVybWlzc2lvbnMgYW5kXG4gKiBsaW1pdGF0aW9ucyB1bmRlciB0aGUgTGljZW5zZS5cbiAqL1xuXG5pbXBvcnQgeyBzY2hlbWEgfSBmcm9tICdAa2JuL2NvbmZpZy1zY2hlbWEnO1xuaW1wb3J0IHsgc2VydmVyRXJyb3IgfSBmcm9tICcuLi8uLi9saWInO1xuaW1wb3J0IHtcbiAgSU5ERVgsXG4gIFJPVVRFX1BBVEgsXG4gIERFRkFVTFRfREFURUZJRUxELFxuICBFU19TQ1JPTExfU0VUVElOR1MsXG4gIE5PX01VTFRJVEVOQU5DWV9URU5BTlQsXG59IGZyb20gJy4uLy4uLy4uLy4uLy4uL2NvbW1vbi9zaWduYWxzL2NvbnN0YW50cyc7XG5cbmV4cG9ydCBjb25zdCBnZXRBbGVydHMgPSAoeyBjbHVzdGVyQ2xpZW50LCBmZXRjaEFsbEZyb21TY3JvbGwsIGxvZ2dlciB9KSA9PiBhc3luYyAoXG4gIGNvbnRleHQsXG4gIHJlcXVlc3QsXG4gIHJlc3BvbnNlXG4pID0+IHtcbiAgdHJ5IHtcbiAgICBjb25zdCB7XG4gICAgICBib2R5OiB7IHF1ZXJ5LCBzb3J0LCBpbmRleCwgc2Nyb2xsIH0sXG4gICAgICBoZWFkZXJzOiB7IHNndGVuYW50IH0sXG4gICAgfSA9IHJlcXVlc3Q7XG5cbiAgICBjb25zdCBvcHRpb25zID0geyBpbmRleCwgc2Nyb2xsIH07XG5cbiAgICBpZiAocXVlcnkgJiYgISFPYmplY3Qua2V5cyhxdWVyeSkubGVuZ3RoKSB7XG4gICAgICAvLyBXZSBkb24ndCBmaWx0ZXIgYWxlcnRzIGJ5IHRlbmFudCBpZiBpdCBpcyBHbG9iYWwgdGVuYW50ICh2YWx1ZSBpcyAnJylcbiAgICAgIGlmIChzZ3RlbmFudCkge1xuICAgICAgICBpZiAoIXF1ZXJ5LmJvb2wubXVzdCkge1xuICAgICAgICAgIHF1ZXJ5LmJvb2wubXVzdCA9IFtdO1xuICAgICAgICB9XG5cbiAgICAgICAgcXVlcnkuYm9vbC5tdXN0LnB1c2goe1xuICAgICAgICAgIHRlcm06IHtcbiAgICAgICAgICAgICd0ZW5hbnQua2V5d29yZCc6IHsgdmFsdWU6IHNndGVuYW50IH0sXG4gICAgICAgICAgfSxcbiAgICAgICAgfSk7XG4gICAgICB9XG5cbiAgICAgIG9wdGlvbnMuYm9keSA9IHsgc29ydCwgcXVlcnkgfTtcbiAgICB9XG5cbiAgICBjb25zdCB7IGJvZHk6IGZpcnN0U2Nyb2xsUmVzcG9uc2UgfSA9IGF3YWl0IGNsdXN0ZXJDbGllbnRcbiAgICAgIC5hc1Njb3BlZChyZXF1ZXN0KVxuICAgICAgLmFzQ3VycmVudFVzZXIuc2VhcmNoKG9wdGlvbnMpO1xuXG4gICAgY29uc3QgaGl0cyA9IGF3YWl0IGZldGNoQWxsRnJvbVNjcm9sbCh7XG4gICAgICBjbHVzdGVyQ2xpZW50LFxuICAgICAgc2Nyb2xsLFxuICAgICAgcmVxdWVzdCxcbiAgICAgIHJlc3BvbnNlOiBmaXJzdFNjcm9sbFJlc3BvbnNlLFxuICAgIH0pO1xuXG4gICAgcmV0dXJuIHJlc3BvbnNlLm9rKHtcbiAgICAgIGJvZHk6IHtcbiAgICAgICAgb2s6IHRydWUsXG4gICAgICAgIHJlc3A6IGhpdHMubWFwKChoKSA9PiAoeyAuLi5oLl9zb3VyY2UsIF9pZDogaC5faWQsIF9pbmRleDogaC5faW5kZXggfSkpLFxuICAgICAgfSxcbiAgICB9KTtcbiAgfSBjYXRjaCAoZXJyKSB7XG4gICAgbG9nZ2VyLmVycm9yKGBnZXRBbGVydHM6ICR7ZXJyLnN0YWNrfWApO1xuICAgIHJldHVybiByZXNwb25zZS5jdXN0b21FcnJvcihzZXJ2ZXJFcnJvcihlcnIpKTtcbiAgfVxufTtcblxuZXhwb3J0IGNvbnN0IGdldEFsZXJ0c1JvdXRlID0gKHsgcm91dGVyLCBjbHVzdGVyQ2xpZW50LCBmZXRjaEFsbEZyb21TY3JvbGwsIGxvZ2dlciB9KSA9PiB7XG4gIHJvdXRlci5wb3N0KFxuICAgIHtcbiAgICAgIHBhdGg6IFJPVVRFX1BBVEguQUxFUlRTLFxuICAgICAgdmFsaWRhdGU6IHtcbiAgICAgICAgaGVhZGVyczogc2NoZW1hLm9iamVjdChcbiAgICAgICAgICB7XG4gICAgICAgICAgICBzZ3RlbmFudDogc2NoZW1hLnN0cmluZyh7IGRlZmF1bHRWYWx1ZTogTk9fTVVMVElURU5BTkNZX1RFTkFOVCB9KSxcbiAgICAgICAgICB9LFxuICAgICAgICAgIHsgdW5rbm93bnM6ICdhbGxvdycgfVxuICAgICAgICApLFxuICAgICAgICBib2R5OiBzY2hlbWEub2JqZWN0KHtcbiAgICAgICAgICBpbmRleDogc2NoZW1hLnN0cmluZyh7IGRlZmF1bHRWYWx1ZTogSU5ERVguQUxFUlRTIH0pLFxuICAgICAgICAgIHNjcm9sbDogc2NoZW1hLnN0cmluZyh7IGRlZmF1bHRWYWx1ZTogRVNfU0NST0xMX1NFVFRJTkdTLktFRVBBTElWRSB9KSxcbiAgICAgICAgICBxdWVyeTogc2NoZW1hLm9iamVjdCh7fSwgeyB1bmtub3duczogJ2FsbG93JyB9KSxcbiAgICAgICAgICBzb3J0OiBzY2hlbWEuYXJyYXlPZihcbiAgICAgICAgICAgIHNjaGVtYS5vYmplY3QoXG4gICAgICAgICAgICAgIHtcbiAgICAgICAgICAgICAgICBbREVGQVVMVF9EQVRFRklFTERdOiBzY2hlbWEuc3RyaW5nKHtcbiAgICAgICAgICAgICAgICAgIHZhbGlkYXRlKHZhbHVlKSB7XG4gICAgICAgICAgICAgICAgICAgIGNvbnN0IGFjY2VwdGVkID0gWydkZXNjJywgJ2FzYyddO1xuICAgICAgICAgICAgICAgICAgICBpZiAoIWFjY2VwdGVkLmluY2x1ZGVzKHZhbHVlKSkge1xuICAgICAgICAgICAgICAgICAgICAgIHJldHVybiBgVGhlIGFjY2VwdGVkIHZhbHVlczogJHthY2NlcHRlZC5qb2luKCcsICcpfWA7XG4gICAgICAgICAgICAgICAgICAgIH1cbiAgICAgICAgICAgICAgICAgIH0sXG4gICAgICAgICAgICAgICAgfSksXG4gICAgICAgICAgICAgIH0sXG4gICAgICAgICAgICAgIHtcbiAgICAgICAgICAgICAgICB1bmtub3duczogJ2FsbG93JyxcbiAgICAgICAgICAgICAgfVxuICAgICAgICAgICAgKSxcbiAgICAgICAgICAgIHtcbiAgICAgICAgICAgICAgZGVmYXVsdFZhbHVlOiB7XG4gICAgICAgICAgICAgICAgW0RFRkFVTFRfREFURUZJRUxEXTogJ2Rlc2MnLFxuICAgICAgICAgICAgICB9LFxuICAgICAgICAgICAgfVxuICAgICAgICAgICksXG4gICAgICAgIH0pLFxuICAgICAgfSxcbiAgICB9LFxuICAgIGdldEFsZXJ0cyh7IGNsdXN0ZXJDbGllbnQsIGZldGNoQWxsRnJvbVNjcm9sbCwgbG9nZ2VyIH0pXG4gICk7XG59O1xuIl0sIm1hcHBpbmdzIjoiOzs7Ozs7QUFnQkEsSUFBQUEsYUFBQSxHQUFBQyxPQUFBO0FBQ0EsSUFBQUMsSUFBQSxHQUFBRCxPQUFBO0FBQ0EsSUFBQUUsVUFBQSxHQUFBRixPQUFBO0FBbEJBO0FBQ0E7QUFDQTtBQUNBO0FBQ0E7QUFDQTtBQUNBO0FBQ0E7QUFDQTtBQUNBO0FBQ0E7QUFDQTtBQUNBO0FBQ0E7QUFDQTs7QUFZTyxNQUFNRyxTQUFTLEdBQUdBLENBQUM7RUFBRUMsYUFBYTtFQUFFQyxrQkFBa0I7RUFBRUM7QUFBTyxDQUFDLEtBQUssT0FDMUVDLE9BQU8sRUFDUEMsT0FBTyxFQUNQQyxRQUFRLEtBQ0w7RUFDSCxJQUFJO0lBQ0YsTUFBTTtNQUNKQyxJQUFJLEVBQUU7UUFBRUMsS0FBSztRQUFFQyxJQUFJO1FBQUVDLEtBQUs7UUFBRUM7TUFBTyxDQUFDO01BQ3BDQyxPQUFPLEVBQUU7UUFBRUM7TUFBUztJQUN0QixDQUFDLEdBQUdSLE9BQU87SUFFWCxNQUFNUyxPQUFPLEdBQUc7TUFBRUosS0FBSztNQUFFQztJQUFPLENBQUM7SUFFakMsSUFBSUgsS0FBSyxJQUFJLENBQUMsQ0FBQ08sTUFBTSxDQUFDQyxJQUFJLENBQUNSLEtBQUssQ0FBQyxDQUFDUyxNQUFNLEVBQUU7TUFDeEM7TUFDQSxJQUFJSixRQUFRLEVBQUU7UUFDWixJQUFJLENBQUNMLEtBQUssQ0FBQ1UsSUFBSSxDQUFDQyxJQUFJLEVBQUU7VUFDcEJYLEtBQUssQ0FBQ1UsSUFBSSxDQUFDQyxJQUFJLEdBQUcsRUFBRTtRQUN0QjtRQUVBWCxLQUFLLENBQUNVLElBQUksQ0FBQ0MsSUFBSSxDQUFDQyxJQUFJLENBQUM7VUFDbkJDLElBQUksRUFBRTtZQUNKLGdCQUFnQixFQUFFO2NBQUVDLEtBQUssRUFBRVQ7WUFBUztVQUN0QztRQUNGLENBQUMsQ0FBQztNQUNKO01BRUFDLE9BQU8sQ0FBQ1AsSUFBSSxHQUFHO1FBQUVFLElBQUk7UUFBRUQ7TUFBTSxDQUFDO0lBQ2hDO0lBRUEsTUFBTTtNQUFFRCxJQUFJLEVBQUVnQjtJQUFvQixDQUFDLEdBQUcsTUFBTXRCLGFBQWEsQ0FDdER1QixRQUFRLENBQUNuQixPQUFPLENBQUMsQ0FDakJvQixhQUFhLENBQUNDLE1BQU0sQ0FBQ1osT0FBTyxDQUFDO0lBRWhDLE1BQU1hLElBQUksR0FBRyxNQUFNekIsa0JBQWtCLENBQUM7TUFDcENELGFBQWE7TUFDYlUsTUFBTTtNQUNOTixPQUFPO01BQ1BDLFFBQVEsRUFBRWlCO0lBQ1osQ0FBQyxDQUFDO0lBRUYsT0FBT2pCLFFBQVEsQ0FBQ3NCLEVBQUUsQ0FBQztNQUNqQnJCLElBQUksRUFBRTtRQUNKcUIsRUFBRSxFQUFFLElBQUk7UUFDUkMsSUFBSSxFQUFFRixJQUFJLENBQUNHLEdBQUcsQ0FBRUMsQ0FBQyxLQUFNO1VBQUUsR0FBR0EsQ0FBQyxDQUFDQyxPQUFPO1VBQUVDLEdBQUcsRUFBRUYsQ0FBQyxDQUFDRSxHQUFHO1VBQUVDLE1BQU0sRUFBRUgsQ0FBQyxDQUFDRztRQUFPLENBQUMsQ0FBQztNQUN4RTtJQUNGLENBQUMsQ0FBQztFQUNKLENBQUMsQ0FBQyxPQUFPQyxHQUFHLEVBQUU7SUFDWmhDLE1BQU0sQ0FBQ2lDLEtBQUssQ0FBQyxjQUFjRCxHQUFHLENBQUNFLEtBQUssRUFBRSxDQUFDO0lBQ3ZDLE9BQU8vQixRQUFRLENBQUNnQyxXQUFXLENBQUMsSUFBQUMsZ0JBQVcsRUFBQ0osR0FBRyxDQUFDLENBQUM7RUFDL0M7QUFDRixDQUFDO0FBQUNLLE9BQUEsQ0FBQXhDLFNBQUEsR0FBQUEsU0FBQTtBQUVLLE1BQU15QyxjQUFjLEdBQUdBLENBQUM7RUFBRUMsTUFBTTtFQUFFekMsYUFBYTtFQUFFQyxrQkFBa0I7RUFBRUM7QUFBTyxDQUFDLEtBQUs7RUFDdkZ1QyxNQUFNLENBQUNDLElBQUksQ0FDVDtJQUNFQyxJQUFJLEVBQUVDLHFCQUFVLENBQUNDLE1BQU07SUFDdkJDLFFBQVEsRUFBRTtNQUNSbkMsT0FBTyxFQUFFb0Msb0JBQU0sQ0FBQ0MsTUFBTSxDQUNwQjtRQUNFcEMsUUFBUSxFQUFFbUMsb0JBQU0sQ0FBQ0UsTUFBTSxDQUFDO1VBQUVDLFlBQVksRUFBRUM7UUFBdUIsQ0FBQztNQUNsRSxDQUFDLEVBQ0Q7UUFBRUMsUUFBUSxFQUFFO01BQVEsQ0FDdEIsQ0FBQztNQUNEOUMsSUFBSSxFQUFFeUMsb0JBQU0sQ0FBQ0MsTUFBTSxDQUFDO1FBQ2xCdkMsS0FBSyxFQUFFc0Msb0JBQU0sQ0FBQ0UsTUFBTSxDQUFDO1VBQUVDLFlBQVksRUFBRUcsZ0JBQUssQ0FBQ1I7UUFBTyxDQUFDLENBQUM7UUFDcERuQyxNQUFNLEVBQUVxQyxvQkFBTSxDQUFDRSxNQUFNLENBQUM7VUFBRUMsWUFBWSxFQUFFSSw2QkFBa0IsQ0FBQ0M7UUFBVSxDQUFDLENBQUM7UUFDckVoRCxLQUFLLEVBQUV3QyxvQkFBTSxDQUFDQyxNQUFNLENBQUMsQ0FBQyxDQUFDLEVBQUU7VUFBRUksUUFBUSxFQUFFO1FBQVEsQ0FBQyxDQUFDO1FBQy9DNUMsSUFBSSxFQUFFdUMsb0JBQU0sQ0FBQ1MsT0FBTyxDQUNsQlQsb0JBQU0sQ0FBQ0MsTUFBTSxDQUNYO1VBQ0UsQ0FBQ1MsNEJBQWlCLEdBQUdWLG9CQUFNLENBQUNFLE1BQU0sQ0FBQztZQUNqQ0gsUUFBUUEsQ0FBQ3pCLEtBQUssRUFBRTtjQUNkLE1BQU1xQyxRQUFRLEdBQUcsQ0FBQyxNQUFNLEVBQUUsS0FBSyxDQUFDO2NBQ2hDLElBQUksQ0FBQ0EsUUFBUSxDQUFDQyxRQUFRLENBQUN0QyxLQUFLLENBQUMsRUFBRTtnQkFDN0IsT0FBTyx3QkFBd0JxQyxRQUFRLENBQUNFLElBQUksQ0FBQyxJQUFJLENBQUMsRUFBRTtjQUN0RDtZQUNGO1VBQ0YsQ0FBQztRQUNILENBQUMsRUFDRDtVQUNFUixRQUFRLEVBQUU7UUFDWixDQUNGLENBQUMsRUFDRDtVQUNFRixZQUFZLEVBQUU7WUFDWixDQUFDTyw0QkFBaUIsR0FBRztVQUN2QjtRQUNGLENBQ0Y7TUFDRixDQUFDO0lBQ0g7RUFDRixDQUFDLEVBQ0QxRCxTQUFTLENBQUM7SUFBRUMsYUFBYTtJQUFFQyxrQkFBa0I7SUFBRUM7RUFBTyxDQUFDLENBQ3pELENBQUM7QUFDSCxDQUFDO0FBQUNxQyxPQUFBLENBQUFDLGNBQUEsR0FBQUEsY0FBQSIsImlnbm9yZUxpc3QiOltdfQ==