"use strict";

Object.defineProperty(exports, "__esModule", {
  value: true
});
exports.OIDC_ROUTES = void 0;
exports.defineRoutes = defineRoutes;
exports.loginHandler = loginHandler;
var _configSchema = require("@kbn/config-schema");
var _sanitize_next_url = require("../../sanitize_next_url");
var _constants = require("../../../../../utils/constants");
/*
 *    Copyright 2021 floragunn GmbH
 *
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 * http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 */

const OIDC_ROUTES = exports.OIDC_ROUTES = {
  LOGIN: `${_constants.APP_ROOT}/auth/oidc/login`
};
function defineRoutes({
  authInstance,
  kibanaCore,
  kibanaConfig,
  logger,
  debugLog,
  searchGuardBackend
}) {
  const config = kibanaConfig;
  const basePath = kibanaCore.http.basePath.serverBasePath;
  const router = kibanaCore.http.createRouter();
  const routesPath = '/auth/oidc/';
  const loginSettings = {
    path: `${_constants.APP_ROOT}${routesPath}login`,
    validate: {
      query: _configSchema.schema.object({
        nextUrl: _configSchema.schema.maybe(_configSchema.schema.string()),
        // it comes from our login page
        next_url: _configSchema.schema.maybe(_configSchema.schema.string()) // it comes from the IDP login page
      }, {
        unknowns: 'allow'
      })
    },
    options: {
      access: 'public',
      authRequired: false
    }
  };
  const loginHandlerOptions = {
    basePath,
    kibanaCore,
    config,
    debugLog,
    authInstance,
    logger,
    searchGuardBackend
  };
  router.get(loginSettings, loginHandler(loginHandlerOptions));

  // Keep a POST route in case the IdP uses POSTs
  router.post(loginSettings, loginHandler(loginHandlerOptions));

  // We want to keep backwards compatibility with /auth/openid requests
  const legacyRoutesPath = '/auth/openid/';
  router.get({
    ...loginSettings,
    path: `${_constants.APP_ROOT}${legacyRoutesPath}login`
  }, loginHandler(loginHandlerOptions));
  router.post({
    ...loginSettings,
    path: `${_constants.APP_ROOT}${legacyRoutesPath}login`
  }, loginHandler(loginHandlerOptions));
} //end module

function loginHandler({
  basePath,
  config,
  authInstance,
  logger,
  searchGuardBackend,
  kibanaCore
}) {
  return async function (context, request, response) {
    const authCode = request.url.searchParams.get('code');
    if (!authCode) {
      return handleAuthRequest({
        request,
        response,
        basePath,
        config,
        searchGuardBackend,
        sessionStorageFactory: authInstance.sessionStorageFactory,
        logger
      });
    }

    // We have an auth code, now we need to try to exchange it for an id_token
    try {
      // Validate the state to make sure that the request was really
      // requested by Kibana in this session
      const sessionCookie = (await authInstance.sessionStorageFactory.asScoped(request).get()) || {};
      const cookieOidc = sessionCookie.oidc;
      if (!cookieOidc) {
        // This seems to happen when we have
        // a) No more session on the IdP
        // and b) We delete our cookie completely.
        // @todo Can we somehow restart the process here?
        throw new Error('OIDC request contained code, but we have no cookie content to compare with');
      }

      // Make sure to clear out what was used for this login request.
      delete sessionCookie.oidc;
      await authInstance.sessionStorageFactory.asScoped(request).set(sessionCookie);
      const credentials = {
        mode: 'oidc',
        sso_result: request.url.href,
        sso_context: cookieOidc.ssoContext,
        id: cookieOidc.authTypeId
      };

      // Authenticate with the token given to us by the IdP
      const authResponse = await authInstance.handleAuthenticate(request, credentials);
      let redirectTo = '/';
      if (authResponse.redirectUri) {
        redirectTo = (0, _sanitize_next_url.sanitizeNextUrl)(authResponse.redirectUri, basePath);
      }

      // All good, redirect to home
      return response.redirected({
        headers: {
          location: redirectTo
        }
      });
    } catch (error) {
      var _error$meta, _error$meta$body, _error$meta3, _error$meta3$body;
      logger.error('Error while trying to authenticate', error.meta ? JSON.stringify(error.meta) : error);
      var headers = {
        location: basePath + '/searchguard/login?err=oidc'
      };
      var cookies = [];
      if ((_error$meta = error.meta) !== null && _error$meta !== void 0 && (_error$meta$body = _error$meta.body) !== null && _error$meta$body !== void 0 && _error$meta$body.error) {
        var _error$meta2, _error$meta2$body;
        cookies.push('sg_err=' + encodeURIComponent((_error$meta2 = error.meta) === null || _error$meta2 === void 0 ? void 0 : (_error$meta2$body = _error$meta2.body) === null || _error$meta2$body === void 0 ? void 0 : _error$meta2$body.error) + "; Path=/");
      } else if (error.message) {
        cookies.push('sg_err=' + encodeURIComponent(error.message) + "; Path=/");
      }
      if ((_error$meta3 = error.meta) !== null && _error$meta3 !== void 0 && (_error$meta3$body = _error$meta3.body) !== null && _error$meta3$body !== void 0 && _error$meta3$body.debug) {
        var _error$meta4, _error$meta4$body;
        cookies.push('sg_dbg=' + encodeURIComponent(JSON.stringify((_error$meta4 = error.meta) === null || _error$meta4 === void 0 ? void 0 : (_error$meta4$body = _error$meta4.body) === null || _error$meta4$body === void 0 ? void 0 : _error$meta4$body.debug)) + "; Path=/");
      }
      if (cookies.length > 0) {
        headers['set-cookie'] = cookies;
      }
      return response.redirected({
        headers
      });
    }
  };
}

/**
 * Handle the first step of the process - obtain an auth code
 * @param request
 * @param response
 * @param sessionStorageFactory
 * @param clientId
 * @param redirectUri
 * @param nonce
 * @param scope
 * @returns {Promise<*>}
 */
async function handleAuthRequest({
  request,
  response,
  basePath,
  config,
  searchGuardBackend,
  sessionStorageFactory,
  logger
}) {
  // Add the nextUrl to the redirect_uri as a parameter. The IDP uses the redirect_uri to redirect the user after successful authentication.
  // For example, it is used to redirect user to the correct dashboard if the user put shared URL in the browser address input before authentication.
  // To make this work, append the wildcard (*) to the valid redirect URI in the IDP configuration, for example
  // https://kibana.example.com:5601/auth/oidc/login*
  let nextUrl = null;
  try {
    if (request.url.searchParams.get('nextUrl') && decodeURIComponent(request.url.searchParams.get('nextUrl')) !== '/') {
      // Do not add the nextUrl to the redirect_uri to avoid the following breaking
      // change for the users that use normal URL to authenticate.
      nextUrl = (0, _sanitize_next_url.sanitizeNextUrl)(request.url.searchParams.get('nextUrl'));
    }
  } catch (error) {
    // We may have received a malformed URL, caught by decodedURIComponent.
    // In this case we just proceed without a nextUrl.
  }
  let authConfig;
  const sessionCookie = (await sessionStorageFactory.asScoped(request).get()) || {};
  // We may have multiple OIDC configurations.
  // The authTypeId may still be in the cookie. This happens when
  // a session token expires and no explicit logout is made. We need
  // this behaviour so that we can "refresh" the credentials from the IdP.
  // (not to be confused with the OIDC refresh flow)

  const requestedAuthTypeId = request.url.searchParams.get('authTypeId') || sessionCookie.authTypeId;
  // Delete this again, otherwise the user won't get back to the login page
  // if trying to access Kibana again
  delete sessionCookie.authTypeId;
  delete sessionCookie.authType;
  const authConfigFinder = requestedAuthTypeId ? config => {
    return config.id === requestedAuthTypeId;
  } : config => {
    return config.method === 'oidc';
  };
  try {
    authConfig = (await searchGuardBackend.getAuthConfig(nextUrl)).auth_methods.find(authConfigFinder);
    if (!authConfig) {
      throw new Error('Auth config not found');
    }
  } catch (error) {
    var _error$meta5, _error$meta5$body, _error$meta7, _error$meta7$body;
    logger.error(`Error when trying to load the configuration for your IdP: ${error.stack}`);
    var headers = {
      location: basePath + '/login?err=oidc_init'
    };
    var cookies = [];
    if ((_error$meta5 = error.meta) !== null && _error$meta5 !== void 0 && (_error$meta5$body = _error$meta5.body) !== null && _error$meta5$body !== void 0 && _error$meta5$body.error) {
      var _error$meta6, _error$meta6$body;
      cookies.push('sg_err=' + encodeURIComponent((_error$meta6 = error.meta) === null || _error$meta6 === void 0 ? void 0 : (_error$meta6$body = _error$meta6.body) === null || _error$meta6$body === void 0 ? void 0 : _error$meta6$body.error) + "; Path=/");
    } else if (error.message) {
      cookies.push('sg_err=' + encodeURIComponent(error.message) + "; Path=/");
    }
    if ((_error$meta7 = error.meta) !== null && _error$meta7 !== void 0 && (_error$meta7$body = _error$meta7.body) !== null && _error$meta7$body !== void 0 && _error$meta7$body.debug) {
      var _error$meta8, _error$meta8$body;
      cookies.push('sg_dbg=' + encodeURIComponent(JSON.stringify((_error$meta8 = error.meta) === null || _error$meta8 === void 0 ? void 0 : (_error$meta8$body = _error$meta8.body) === null || _error$meta8$body === void 0 ? void 0 : _error$meta8$body.debug)) + "; Path=/");
    }
    if (cookies.length > 0) {
      headers['set-cookie'] = cookies;
    }
    return response.redirected({
      headers
    });
  }
  sessionCookie.oidc = {
    ssoContext: authConfig.sso_context,
    authTypeId: authConfig.id || null,
    query: {}
  };
  await sessionStorageFactory.asScoped(request).set(sessionCookie);
  return response.redirected({
    headers: {
      location: authConfig.sso_location
    }
  });
}
//# sourceMappingURL=data:application/json;charset=utf-8;base64,eyJ2ZXJzaW9uIjozLCJuYW1lcyI6WyJfY29uZmlnU2NoZW1hIiwicmVxdWlyZSIsIl9zYW5pdGl6ZV9uZXh0X3VybCIsIl9jb25zdGFudHMiLCJPSURDX1JPVVRFUyIsImV4cG9ydHMiLCJMT0dJTiIsIkFQUF9ST09UIiwiZGVmaW5lUm91dGVzIiwiYXV0aEluc3RhbmNlIiwia2liYW5hQ29yZSIsImtpYmFuYUNvbmZpZyIsImxvZ2dlciIsImRlYnVnTG9nIiwic2VhcmNoR3VhcmRCYWNrZW5kIiwiY29uZmlnIiwiYmFzZVBhdGgiLCJodHRwIiwic2VydmVyQmFzZVBhdGgiLCJyb3V0ZXIiLCJjcmVhdGVSb3V0ZXIiLCJyb3V0ZXNQYXRoIiwibG9naW5TZXR0aW5ncyIsInBhdGgiLCJ2YWxpZGF0ZSIsInF1ZXJ5Iiwic2NoZW1hIiwib2JqZWN0IiwibmV4dFVybCIsIm1heWJlIiwic3RyaW5nIiwibmV4dF91cmwiLCJ1bmtub3ducyIsIm9wdGlvbnMiLCJhY2Nlc3MiLCJhdXRoUmVxdWlyZWQiLCJsb2dpbkhhbmRsZXJPcHRpb25zIiwiZ2V0IiwibG9naW5IYW5kbGVyIiwicG9zdCIsImxlZ2FjeVJvdXRlc1BhdGgiLCJjb250ZXh0IiwicmVxdWVzdCIsInJlc3BvbnNlIiwiYXV0aENvZGUiLCJ1cmwiLCJzZWFyY2hQYXJhbXMiLCJoYW5kbGVBdXRoUmVxdWVzdCIsInNlc3Npb25TdG9yYWdlRmFjdG9yeSIsInNlc3Npb25Db29raWUiLCJhc1Njb3BlZCIsImNvb2tpZU9pZGMiLCJvaWRjIiwiRXJyb3IiLCJzZXQiLCJjcmVkZW50aWFscyIsIm1vZGUiLCJzc29fcmVzdWx0IiwiaHJlZiIsInNzb19jb250ZXh0Iiwic3NvQ29udGV4dCIsImlkIiwiYXV0aFR5cGVJZCIsImF1dGhSZXNwb25zZSIsImhhbmRsZUF1dGhlbnRpY2F0ZSIsInJlZGlyZWN0VG8iLCJyZWRpcmVjdFVyaSIsInNhbml0aXplTmV4dFVybCIsInJlZGlyZWN0ZWQiLCJoZWFkZXJzIiwibG9jYXRpb24iLCJlcnJvciIsIl9lcnJvciRtZXRhIiwiX2Vycm9yJG1ldGEkYm9keSIsIl9lcnJvciRtZXRhMyIsIl9lcnJvciRtZXRhMyRib2R5IiwibWV0YSIsIkpTT04iLCJzdHJpbmdpZnkiLCJjb29raWVzIiwiYm9keSIsIl9lcnJvciRtZXRhMiIsIl9lcnJvciRtZXRhMiRib2R5IiwicHVzaCIsImVuY29kZVVSSUNvbXBvbmVudCIsIm1lc3NhZ2UiLCJkZWJ1ZyIsIl9lcnJvciRtZXRhNCIsIl9lcnJvciRtZXRhNCRib2R5IiwibGVuZ3RoIiwiZGVjb2RlVVJJQ29tcG9uZW50IiwiYXV0aENvbmZpZyIsInJlcXVlc3RlZEF1dGhUeXBlSWQiLCJhdXRoVHlwZSIsImF1dGhDb25maWdGaW5kZXIiLCJtZXRob2QiLCJnZXRBdXRoQ29uZmlnIiwiYXV0aF9tZXRob2RzIiwiZmluZCIsIl9lcnJvciRtZXRhNSIsIl9lcnJvciRtZXRhNSRib2R5IiwiX2Vycm9yJG1ldGE3IiwiX2Vycm9yJG1ldGE3JGJvZHkiLCJzdGFjayIsIl9lcnJvciRtZXRhNiIsIl9lcnJvciRtZXRhNiRib2R5IiwiX2Vycm9yJG1ldGE4IiwiX2Vycm9yJG1ldGE4JGJvZHkiLCJzc29fbG9jYXRpb24iXSwic291cmNlcyI6WyJyb3V0ZXMuanMiXSwic291cmNlc0NvbnRlbnQiOlsiLypcbiAqICAgIENvcHlyaWdodCAyMDIxIGZsb3JhZ3VubiBHbWJIXG4gKlxuICogTGljZW5zZWQgdW5kZXIgdGhlIEFwYWNoZSBMaWNlbnNlLCBWZXJzaW9uIDIuMCAodGhlIFwiTGljZW5zZVwiKTtcbiAqIHlvdSBtYXkgbm90IHVzZSB0aGlzIGZpbGUgZXhjZXB0IGluIGNvbXBsaWFuY2Ugd2l0aCB0aGUgTGljZW5zZS5cbiAqIFlvdSBtYXkgb2J0YWluIGEgY29weSBvZiB0aGUgTGljZW5zZSBhdFxuICpcbiAqIGh0dHA6Ly93d3cuYXBhY2hlLm9yZy9saWNlbnNlcy9MSUNFTlNFLTIuMFxuICpcbiAqIFVubGVzcyByZXF1aXJlZCBieSBhcHBsaWNhYmxlIGxhdyBvciBhZ3JlZWQgdG8gaW4gd3JpdGluZywgc29mdHdhcmVcbiAqIGRpc3RyaWJ1dGVkIHVuZGVyIHRoZSBMaWNlbnNlIGlzIGRpc3RyaWJ1dGVkIG9uIGFuIFwiQVMgSVNcIiBCQVNJUyxcbiAqIFdJVEhPVVQgV0FSUkFOVElFUyBPUiBDT05ESVRJT05TIE9GIEFOWSBLSU5ELCBlaXRoZXIgZXhwcmVzcyBvciBpbXBsaWVkLlxuICogU2VlIHRoZSBMaWNlbnNlIGZvciB0aGUgc3BlY2lmaWMgbGFuZ3VhZ2UgZ292ZXJuaW5nIHBlcm1pc3Npb25zIGFuZFxuICogbGltaXRhdGlvbnMgdW5kZXIgdGhlIExpY2Vuc2UuXG4gKi9cblxuaW1wb3J0IHsgc2NoZW1hIH0gZnJvbSAnQGtibi9jb25maWctc2NoZW1hJztcbmltcG9ydCB7IHNhbml0aXplTmV4dFVybCB9IGZyb20gJy4uLy4uL3Nhbml0aXplX25leHRfdXJsJztcbmltcG9ydCB7IEFQUF9ST09UIH0gZnJvbSAnLi4vLi4vLi4vLi4vLi4vdXRpbHMvY29uc3RhbnRzJztcblxuZXhwb3J0IGNvbnN0IE9JRENfUk9VVEVTID0ge1xuICBMT0dJTjogYCR7QVBQX1JPT1R9L2F1dGgvb2lkYy9sb2dpbmAsXG59O1xuXG5leHBvcnQgZnVuY3Rpb24gZGVmaW5lUm91dGVzKHtcbiAgYXV0aEluc3RhbmNlLFxuICBraWJhbmFDb3JlLFxuICBraWJhbmFDb25maWcsXG4gIGxvZ2dlcixcbiAgZGVidWdMb2csXG4gIHNlYXJjaEd1YXJkQmFja2VuZCxcbn0pIHtcbiAgY29uc3QgY29uZmlnID0ga2liYW5hQ29uZmlnO1xuICBjb25zdCBiYXNlUGF0aCA9IGtpYmFuYUNvcmUuaHR0cC5iYXNlUGF0aC5zZXJ2ZXJCYXNlUGF0aDtcbiAgY29uc3Qgcm91dGVyID0ga2liYW5hQ29yZS5odHRwLmNyZWF0ZVJvdXRlcigpO1xuICBjb25zdCByb3V0ZXNQYXRoID0gJy9hdXRoL29pZGMvJztcblxuICBjb25zdCBsb2dpblNldHRpbmdzID0ge1xuICAgIHBhdGg6IGAke0FQUF9ST09UfSR7cm91dGVzUGF0aH1sb2dpbmAsXG4gICAgdmFsaWRhdGU6IHtcbiAgICAgIHF1ZXJ5OiBzY2hlbWEub2JqZWN0KFxuICAgICAgICB7XG4gICAgICAgICAgbmV4dFVybDogc2NoZW1hLm1heWJlKHNjaGVtYS5zdHJpbmcoKSksIC8vIGl0IGNvbWVzIGZyb20gb3VyIGxvZ2luIHBhZ2VcbiAgICAgICAgICBuZXh0X3VybDogc2NoZW1hLm1heWJlKHNjaGVtYS5zdHJpbmcoKSksIC8vIGl0IGNvbWVzIGZyb20gdGhlIElEUCBsb2dpbiBwYWdlXG4gICAgICAgIH0sXG4gICAgICAgIHsgdW5rbm93bnM6ICdhbGxvdycgfVxuICAgICAgKSxcbiAgICB9LFxuICAgIG9wdGlvbnM6IHtcbiAgICAgIGFjY2VzczogJ3B1YmxpYycsXG4gICAgICBhdXRoUmVxdWlyZWQ6IGZhbHNlLFxuICAgIH0sXG4gIH07XG5cbiAgY29uc3QgbG9naW5IYW5kbGVyT3B0aW9ucyA9IHtcbiAgICBiYXNlUGF0aCxcbiAgICBraWJhbmFDb3JlLFxuICAgIGNvbmZpZyxcbiAgICBkZWJ1Z0xvZyxcbiAgICBhdXRoSW5zdGFuY2UsXG4gICAgbG9nZ2VyLFxuICAgIHNlYXJjaEd1YXJkQmFja2VuZCxcbiAgfVxuXG4gIHJvdXRlci5nZXQoXG4gICAgbG9naW5TZXR0aW5ncyxcbiAgICBsb2dpbkhhbmRsZXIobG9naW5IYW5kbGVyT3B0aW9ucylcbiAgKTtcblxuICAvLyBLZWVwIGEgUE9TVCByb3V0ZSBpbiBjYXNlIHRoZSBJZFAgdXNlcyBQT1NUc1xuICByb3V0ZXIucG9zdChcbiAgICBsb2dpblNldHRpbmdzLFxuICAgIGxvZ2luSGFuZGxlcihsb2dpbkhhbmRsZXJPcHRpb25zKVxuICApO1xuXG5cbiAgLy8gV2Ugd2FudCB0byBrZWVwIGJhY2t3YXJkcyBjb21wYXRpYmlsaXR5IHdpdGggL2F1dGgvb3BlbmlkIHJlcXVlc3RzXG4gIGNvbnN0IGxlZ2FjeVJvdXRlc1BhdGggPSAnL2F1dGgvb3BlbmlkLyc7XG4gIFxuICByb3V0ZXIuZ2V0KFxuICAgIHtcbiAgICAgIC4uLmxvZ2luU2V0dGluZ3MsXG4gICAgICBwYXRoOiBgJHtBUFBfUk9PVH0ke2xlZ2FjeVJvdXRlc1BhdGh9bG9naW5gXG4gICAgfSxcbiAgICBsb2dpbkhhbmRsZXIobG9naW5IYW5kbGVyT3B0aW9ucylcbiAgKVxuXG4gIHJvdXRlci5wb3N0KFxuICAgIHtcbiAgICAgIC4uLmxvZ2luU2V0dGluZ3MsXG4gICAgICBwYXRoOiBgJHtBUFBfUk9PVH0ke2xlZ2FjeVJvdXRlc1BhdGh9bG9naW5gXG4gICAgfSxcbiAgICBsb2dpbkhhbmRsZXIobG9naW5IYW5kbGVyT3B0aW9ucylcbiAgKVxufSAvL2VuZCBtb2R1bGVcblxuZXhwb3J0IGZ1bmN0aW9uIGxvZ2luSGFuZGxlcih7IGJhc2VQYXRoLCBjb25maWcsIGF1dGhJbnN0YW5jZSwgbG9nZ2VyLCBzZWFyY2hHdWFyZEJhY2tlbmQsIGtpYmFuYUNvcmUgfSkge1xuICByZXR1cm4gYXN5bmMgZnVuY3Rpb24gKGNvbnRleHQsIHJlcXVlc3QsIHJlc3BvbnNlKSB7XG4gICAgY29uc3QgYXV0aENvZGUgPSByZXF1ZXN0LnVybC5zZWFyY2hQYXJhbXMuZ2V0KCdjb2RlJyk7XG5cbiAgICBpZiAoIWF1dGhDb2RlKSB7XG4gICAgICByZXR1cm4gaGFuZGxlQXV0aFJlcXVlc3Qoe1xuICAgICAgICByZXF1ZXN0LFxuICAgICAgICByZXNwb25zZSxcbiAgICAgICAgYmFzZVBhdGgsXG4gICAgICAgIGNvbmZpZyxcbiAgICAgICAgc2VhcmNoR3VhcmRCYWNrZW5kLFxuICAgICAgICBzZXNzaW9uU3RvcmFnZUZhY3Rvcnk6IGF1dGhJbnN0YW5jZS5zZXNzaW9uU3RvcmFnZUZhY3RvcnksXG4gICAgICAgIGxvZ2dlcixcbiAgICAgIH0pO1xuICAgIH1cblxuICAgIC8vIFdlIGhhdmUgYW4gYXV0aCBjb2RlLCBub3cgd2UgbmVlZCB0byB0cnkgdG8gZXhjaGFuZ2UgaXQgZm9yIGFuIGlkX3Rva2VuXG4gICAgdHJ5IHtcbiAgICAgIC8vIFZhbGlkYXRlIHRoZSBzdGF0ZSB0byBtYWtlIHN1cmUgdGhhdCB0aGUgcmVxdWVzdCB3YXMgcmVhbGx5XG4gICAgICAvLyByZXF1ZXN0ZWQgYnkgS2liYW5hIGluIHRoaXMgc2Vzc2lvblxuICAgICAgY29uc3Qgc2Vzc2lvbkNvb2tpZSA9XG4gICAgICAgIChhd2FpdCBhdXRoSW5zdGFuY2Uuc2Vzc2lvblN0b3JhZ2VGYWN0b3J5LmFzU2NvcGVkKHJlcXVlc3QpLmdldCgpKSB8fCB7fTtcblxuICAgICAgY29uc3QgY29va2llT2lkYyA9IHNlc3Npb25Db29raWUub2lkYztcblxuICAgICAgaWYgKCFjb29raWVPaWRjKSB7XG4gICAgICAgIC8vIFRoaXMgc2VlbXMgdG8gaGFwcGVuIHdoZW4gd2UgaGF2ZVxuICAgICAgICAvLyBhKSBObyBtb3JlIHNlc3Npb24gb24gdGhlIElkUFxuICAgICAgICAvLyBhbmQgYikgV2UgZGVsZXRlIG91ciBjb29raWUgY29tcGxldGVseS5cbiAgICAgICAgLy8gQHRvZG8gQ2FuIHdlIHNvbWVob3cgcmVzdGFydCB0aGUgcHJvY2VzcyBoZXJlP1xuICAgICAgICB0aHJvdyBuZXcgRXJyb3IoXG4gICAgICAgICAgJ09JREMgcmVxdWVzdCBjb250YWluZWQgY29kZSwgYnV0IHdlIGhhdmUgbm8gY29va2llIGNvbnRlbnQgdG8gY29tcGFyZSB3aXRoJ1xuICAgICAgICApO1xuICAgICAgfVxuXG4gICAgICAvLyBNYWtlIHN1cmUgdG8gY2xlYXIgb3V0IHdoYXQgd2FzIHVzZWQgZm9yIHRoaXMgbG9naW4gcmVxdWVzdC5cbiAgICAgIGRlbGV0ZSBzZXNzaW9uQ29va2llLm9pZGM7XG4gICAgICBhd2FpdCBhdXRoSW5zdGFuY2Uuc2Vzc2lvblN0b3JhZ2VGYWN0b3J5LmFzU2NvcGVkKHJlcXVlc3QpLnNldChzZXNzaW9uQ29va2llKTtcblxuICAgICAgY29uc3QgY3JlZGVudGlhbHMgPSB7XG4gICAgICAgIG1vZGU6ICdvaWRjJyxcbiAgICAgICAgc3NvX3Jlc3VsdDogcmVxdWVzdC51cmwuaHJlZixcbiAgICAgICAgc3NvX2NvbnRleHQ6IGNvb2tpZU9pZGMuc3NvQ29udGV4dCxcbiAgICAgICAgaWQ6IGNvb2tpZU9pZGMuYXV0aFR5cGVJZCxcbiAgICAgIH07XG5cbiAgICAgIC8vIEF1dGhlbnRpY2F0ZSB3aXRoIHRoZSB0b2tlbiBnaXZlbiB0byB1cyBieSB0aGUgSWRQXG4gICAgICBjb25zdCBhdXRoUmVzcG9uc2UgPSBhd2FpdCBhdXRoSW5zdGFuY2UuaGFuZGxlQXV0aGVudGljYXRlKHJlcXVlc3QsIGNyZWRlbnRpYWxzKTtcblxuICAgICAgbGV0IHJlZGlyZWN0VG8gPSAnLyc7XG4gICAgICBpZiAoYXV0aFJlc3BvbnNlLnJlZGlyZWN0VXJpKSB7XG4gICAgICAgIHJlZGlyZWN0VG8gPSBzYW5pdGl6ZU5leHRVcmwoYXV0aFJlc3BvbnNlLnJlZGlyZWN0VXJpLCBiYXNlUGF0aCk7XG4gICAgICB9XG5cbiAgICAgIC8vIEFsbCBnb29kLCByZWRpcmVjdCB0byBob21lXG4gICAgICByZXR1cm4gcmVzcG9uc2UucmVkaXJlY3RlZCh7XG4gICAgICAgIGhlYWRlcnM6IHtcbiAgICAgICAgICBsb2NhdGlvbjogcmVkaXJlY3RUbyxcbiAgICAgICAgfSxcbiAgICAgIH0pO1xuICAgIH0gY2F0Y2ggKGVycm9yKSB7XG4gICAgICBsb2dnZXIuZXJyb3IoJ0Vycm9yIHdoaWxlIHRyeWluZyB0byBhdXRoZW50aWNhdGUnLCBlcnJvci5tZXRhID8gSlNPTi5zdHJpbmdpZnkoZXJyb3IubWV0YSkgOiBlcnJvcik7XG5cblx0ICB2YXIgaGVhZGVycyA9IHtcblx0XHRsb2NhdGlvbjogYmFzZVBhdGggKyAnL3NlYXJjaGd1YXJkL2xvZ2luP2Vycj1vaWRjJyxcblx0ICB9O1xuXG4gICAgICB2YXIgY29va2llcyA9IFtdO1xuXG4gICAgICBpZiAoZXJyb3IubWV0YT8uYm9keT8uZXJyb3IpIHtcblx0ICAgICBjb29raWVzLnB1c2goJ3NnX2Vycj0nICsgZW5jb2RlVVJJQ29tcG9uZW50KGVycm9yLm1ldGE/LmJvZHk/LmVycm9yKSArIFwiOyBQYXRoPS9cIik7XG5cdCAgfSBlbHNlIGlmIChlcnJvci5tZXNzYWdlKSB7XG5cdCAgICAgY29va2llcy5wdXNoKCdzZ19lcnI9JyArIGVuY29kZVVSSUNvbXBvbmVudChlcnJvci5tZXNzYWdlKSArIFwiOyBQYXRoPS9cIik7XG5cdCAgfVxuXG4gICAgICBpZiAoZXJyb3IubWV0YT8uYm9keT8uZGVidWcpIHtcblx0ICAgICBjb29raWVzLnB1c2goJ3NnX2RiZz0nICsgZW5jb2RlVVJJQ29tcG9uZW50KEpTT04uc3RyaW5naWZ5KGVycm9yLm1ldGE/LmJvZHk/LmRlYnVnKSkgKyBcIjsgUGF0aD0vXCIpO1xuXHQgIH1cblxuXHQgIGlmIChjb29raWVzLmxlbmd0aCA+IDApIHtcbiAgICAgICAgaGVhZGVyc1snc2V0LWNvb2tpZSddID0gY29va2llcztcdFx0XG4gICAgICB9XG5cbiAgICAgIHJldHVybiByZXNwb25zZS5yZWRpcmVjdGVkKHtcbiAgICAgICAgaGVhZGVycyxcbiAgICAgIH0pO1xuICAgIH1cbiAgfTtcbn1cblxuLyoqXG4gKiBIYW5kbGUgdGhlIGZpcnN0IHN0ZXAgb2YgdGhlIHByb2Nlc3MgLSBvYnRhaW4gYW4gYXV0aCBjb2RlXG4gKiBAcGFyYW0gcmVxdWVzdFxuICogQHBhcmFtIHJlc3BvbnNlXG4gKiBAcGFyYW0gc2Vzc2lvblN0b3JhZ2VGYWN0b3J5XG4gKiBAcGFyYW0gY2xpZW50SWRcbiAqIEBwYXJhbSByZWRpcmVjdFVyaVxuICogQHBhcmFtIG5vbmNlXG4gKiBAcGFyYW0gc2NvcGVcbiAqIEByZXR1cm5zIHtQcm9taXNlPCo+fVxuICovXG5hc3luYyBmdW5jdGlvbiBoYW5kbGVBdXRoUmVxdWVzdCh7XG4gIHJlcXVlc3QsXG4gIHJlc3BvbnNlLFxuICBiYXNlUGF0aCxcbiAgY29uZmlnLFxuICBzZWFyY2hHdWFyZEJhY2tlbmQsXG4gIHNlc3Npb25TdG9yYWdlRmFjdG9yeSxcbiAgbG9nZ2VyLFxufSkge1xuICAvLyBBZGQgdGhlIG5leHRVcmwgdG8gdGhlIHJlZGlyZWN0X3VyaSBhcyBhIHBhcmFtZXRlci4gVGhlIElEUCB1c2VzIHRoZSByZWRpcmVjdF91cmkgdG8gcmVkaXJlY3QgdGhlIHVzZXIgYWZ0ZXIgc3VjY2Vzc2Z1bCBhdXRoZW50aWNhdGlvbi5cbiAgLy8gRm9yIGV4YW1wbGUsIGl0IGlzIHVzZWQgdG8gcmVkaXJlY3QgdXNlciB0byB0aGUgY29ycmVjdCBkYXNoYm9hcmQgaWYgdGhlIHVzZXIgcHV0IHNoYXJlZCBVUkwgaW4gdGhlIGJyb3dzZXIgYWRkcmVzcyBpbnB1dCBiZWZvcmUgYXV0aGVudGljYXRpb24uXG4gIC8vIFRvIG1ha2UgdGhpcyB3b3JrLCBhcHBlbmQgdGhlIHdpbGRjYXJkICgqKSB0byB0aGUgdmFsaWQgcmVkaXJlY3QgVVJJIGluIHRoZSBJRFAgY29uZmlndXJhdGlvbiwgZm9yIGV4YW1wbGVcbiAgLy8gaHR0cHM6Ly9raWJhbmEuZXhhbXBsZS5jb206NTYwMS9hdXRoL29pZGMvbG9naW4qXG4gIGxldCBuZXh0VXJsID0gbnVsbDtcbiAgdHJ5IHtcbiAgICBpZiAocmVxdWVzdC51cmwuc2VhcmNoUGFyYW1zLmdldCgnbmV4dFVybCcpICYmIGRlY29kZVVSSUNvbXBvbmVudChyZXF1ZXN0LnVybC5zZWFyY2hQYXJhbXMuZ2V0KCduZXh0VXJsJykpICE9PSAnLycpIHtcbiAgICAgIC8vIERvIG5vdCBhZGQgdGhlIG5leHRVcmwgdG8gdGhlIHJlZGlyZWN0X3VyaSB0byBhdm9pZCB0aGUgZm9sbG93aW5nIGJyZWFraW5nXG4gICAgICAvLyBjaGFuZ2UgZm9yIHRoZSB1c2VycyB0aGF0IHVzZSBub3JtYWwgVVJMIHRvIGF1dGhlbnRpY2F0ZS5cbiAgICAgIG5leHRVcmwgPSBzYW5pdGl6ZU5leHRVcmwocmVxdWVzdC51cmwuc2VhcmNoUGFyYW1zLmdldCgnbmV4dFVybCcpKTtcbiAgICB9XG4gIH0gY2F0Y2ggKGVycm9yKSB7XG4gICAgLy8gV2UgbWF5IGhhdmUgcmVjZWl2ZWQgYSBtYWxmb3JtZWQgVVJMLCBjYXVnaHQgYnkgZGVjb2RlZFVSSUNvbXBvbmVudC5cbiAgICAvLyBJbiB0aGlzIGNhc2Ugd2UganVzdCBwcm9jZWVkIHdpdGhvdXQgYSBuZXh0VXJsLlxuICB9XG5cbiAgbGV0IGF1dGhDb25maWc7XG4gIGNvbnN0IHNlc3Npb25Db29raWUgPSAoYXdhaXQgc2Vzc2lvblN0b3JhZ2VGYWN0b3J5LmFzU2NvcGVkKHJlcXVlc3QpLmdldCgpKSB8fCB7fTtcbiAgLy8gV2UgbWF5IGhhdmUgbXVsdGlwbGUgT0lEQyBjb25maWd1cmF0aW9ucy5cbiAgLy8gVGhlIGF1dGhUeXBlSWQgbWF5IHN0aWxsIGJlIGluIHRoZSBjb29raWUuIFRoaXMgaGFwcGVucyB3aGVuXG4gIC8vIGEgc2Vzc2lvbiB0b2tlbiBleHBpcmVzIGFuZCBubyBleHBsaWNpdCBsb2dvdXQgaXMgbWFkZS4gV2UgbmVlZFxuICAvLyB0aGlzIGJlaGF2aW91ciBzbyB0aGF0IHdlIGNhbiBcInJlZnJlc2hcIiB0aGUgY3JlZGVudGlhbHMgZnJvbSB0aGUgSWRQLlxuICAvLyAobm90IHRvIGJlIGNvbmZ1c2VkIHdpdGggdGhlIE9JREMgcmVmcmVzaCBmbG93KVxuXG4gIGNvbnN0IHJlcXVlc3RlZEF1dGhUeXBlSWQgPSByZXF1ZXN0LnVybC5zZWFyY2hQYXJhbXMuZ2V0KCdhdXRoVHlwZUlkJykgfHwgc2Vzc2lvbkNvb2tpZS5hdXRoVHlwZUlkO1xuICAvLyBEZWxldGUgdGhpcyBhZ2Fpbiwgb3RoZXJ3aXNlIHRoZSB1c2VyIHdvbid0IGdldCBiYWNrIHRvIHRoZSBsb2dpbiBwYWdlXG4gIC8vIGlmIHRyeWluZyB0byBhY2Nlc3MgS2liYW5hIGFnYWluXG4gIGRlbGV0ZSBzZXNzaW9uQ29va2llLmF1dGhUeXBlSWQ7XG4gIGRlbGV0ZSBzZXNzaW9uQ29va2llLmF1dGhUeXBlO1xuXG4gIGNvbnN0IGF1dGhDb25maWdGaW5kZXIgPSByZXF1ZXN0ZWRBdXRoVHlwZUlkXG4gICAgPyAoY29uZmlnKSA9PiB7XG4gICAgICAgIHJldHVybiBjb25maWcuaWQgPT09IHJlcXVlc3RlZEF1dGhUeXBlSWQ7XG4gICAgICB9XG4gICAgOiAoY29uZmlnKSA9PiB7XG4gICAgICAgIHJldHVybiBjb25maWcubWV0aG9kID09PSAnb2lkYyc7XG4gICAgICB9O1xuXG4gIHRyeSB7XG4gICAgYXV0aENvbmZpZyA9IChcbiAgICAgIGF3YWl0IHNlYXJjaEd1YXJkQmFja2VuZC5nZXRBdXRoQ29uZmlnKG5leHRVcmwpXG4gICAgKS5hdXRoX21ldGhvZHMuZmluZChhdXRoQ29uZmlnRmluZGVyKTtcblxuICAgIGlmICghYXV0aENvbmZpZykge1xuICAgICAgdGhyb3cgbmV3IEVycm9yKCdBdXRoIGNvbmZpZyBub3QgZm91bmQnKTtcbiAgICB9XG4gIH0gY2F0Y2ggKGVycm9yKSB7XG4gICAgbG9nZ2VyLmVycm9yKGBFcnJvciB3aGVuIHRyeWluZyB0byBsb2FkIHRoZSBjb25maWd1cmF0aW9uIGZvciB5b3VyIElkUDogJHtlcnJvci5zdGFja31gKTtcblxuXHR2YXIgaGVhZGVycyA9IHtcblx0XHRsb2NhdGlvbjogYmFzZVBhdGggKyAnL2xvZ2luP2Vycj1vaWRjX2luaXQnLFxuXHR9O1xuXG4gICAgdmFyIGNvb2tpZXMgPSBbXTtcblxuICAgIGlmIChlcnJvci5tZXRhPy5ib2R5Py5lcnJvcikge1xuXHQgICBjb29raWVzLnB1c2goJ3NnX2Vycj0nICsgZW5jb2RlVVJJQ29tcG9uZW50KGVycm9yLm1ldGE/LmJvZHk/LmVycm9yKSArIFwiOyBQYXRoPS9cIik7XG5cdH0gZWxzZSBpZiAoZXJyb3IubWVzc2FnZSkge1xuXHQgICBjb29raWVzLnB1c2goJ3NnX2Vycj0nICsgZW5jb2RlVVJJQ29tcG9uZW50KGVycm9yLm1lc3NhZ2UpICsgXCI7IFBhdGg9L1wiKTtcblx0fVxuXG4gICAgaWYgKGVycm9yLm1ldGE/LmJvZHk/LmRlYnVnKSB7XG5cdCAgIGNvb2tpZXMucHVzaCgnc2dfZGJnPScgKyBlbmNvZGVVUklDb21wb25lbnQoSlNPTi5zdHJpbmdpZnkoZXJyb3IubWV0YT8uYm9keT8uZGVidWcpKSArIFwiOyBQYXRoPS9cIik7XG5cdH1cblxuXHRpZiAoY29va2llcy5sZW5ndGggPiAwKSB7XG4gICAgICBoZWFkZXJzWydzZXQtY29va2llJ10gPSBjb29raWVzO1x0XHRcbiAgICB9XG5cbiAgICByZXR1cm4gcmVzcG9uc2UucmVkaXJlY3RlZCh7XG4gICAgICAgIGhlYWRlcnMsXG4gICAgfSk7XG4gIH1cblxuICBzZXNzaW9uQ29va2llLm9pZGMgPSB7IHNzb0NvbnRleHQ6IGF1dGhDb25maWcuc3NvX2NvbnRleHQsIGF1dGhUeXBlSWQ6IGF1dGhDb25maWcuaWQgfHwgbnVsbCwgcXVlcnk6IHt9IH07XG4gIGF3YWl0IHNlc3Npb25TdG9yYWdlRmFjdG9yeS5hc1Njb3BlZChyZXF1ZXN0KS5zZXQoc2Vzc2lvbkNvb2tpZSk7XG5cbiAgcmV0dXJuIHJlc3BvbnNlLnJlZGlyZWN0ZWQoe1xuICAgIGhlYWRlcnM6IHtcbiAgICAgIGxvY2F0aW9uOiBhdXRoQ29uZmlnLnNzb19sb2NhdGlvbixcbiAgICB9LFxuICB9KTtcbn1cbiJdLCJtYXBwaW5ncyI6Ijs7Ozs7Ozs7QUFnQkEsSUFBQUEsYUFBQSxHQUFBQyxPQUFBO0FBQ0EsSUFBQUMsa0JBQUEsR0FBQUQsT0FBQTtBQUNBLElBQUFFLFVBQUEsR0FBQUYsT0FBQTtBQWxCQTtBQUNBO0FBQ0E7QUFDQTtBQUNBO0FBQ0E7QUFDQTtBQUNBO0FBQ0E7QUFDQTtBQUNBO0FBQ0E7QUFDQTtBQUNBO0FBQ0E7O0FBTU8sTUFBTUcsV0FBVyxHQUFBQyxPQUFBLENBQUFELFdBQUEsR0FBRztFQUN6QkUsS0FBSyxFQUFFLEdBQUdDLG1CQUFRO0FBQ3BCLENBQUM7QUFFTSxTQUFTQyxZQUFZQSxDQUFDO0VBQzNCQyxZQUFZO0VBQ1pDLFVBQVU7RUFDVkMsWUFBWTtFQUNaQyxNQUFNO0VBQ05DLFFBQVE7RUFDUkM7QUFDRixDQUFDLEVBQUU7RUFDRCxNQUFNQyxNQUFNLEdBQUdKLFlBQVk7RUFDM0IsTUFBTUssUUFBUSxHQUFHTixVQUFVLENBQUNPLElBQUksQ0FBQ0QsUUFBUSxDQUFDRSxjQUFjO0VBQ3hELE1BQU1DLE1BQU0sR0FBR1QsVUFBVSxDQUFDTyxJQUFJLENBQUNHLFlBQVksQ0FBQyxDQUFDO0VBQzdDLE1BQU1DLFVBQVUsR0FBRyxhQUFhO0VBRWhDLE1BQU1DLGFBQWEsR0FBRztJQUNwQkMsSUFBSSxFQUFFLEdBQUdoQixtQkFBUSxHQUFHYyxVQUFVLE9BQU87SUFDckNHLFFBQVEsRUFBRTtNQUNSQyxLQUFLLEVBQUVDLG9CQUFNLENBQUNDLE1BQU0sQ0FDbEI7UUFDRUMsT0FBTyxFQUFFRixvQkFBTSxDQUFDRyxLQUFLLENBQUNILG9CQUFNLENBQUNJLE1BQU0sQ0FBQyxDQUFDLENBQUM7UUFBRTtRQUN4Q0MsUUFBUSxFQUFFTCxvQkFBTSxDQUFDRyxLQUFLLENBQUNILG9CQUFNLENBQUNJLE1BQU0sQ0FBQyxDQUFDLENBQUMsQ0FBRTtNQUMzQyxDQUFDLEVBQ0Q7UUFBRUUsUUFBUSxFQUFFO01BQVEsQ0FDdEI7SUFDRixDQUFDO0lBQ0RDLE9BQU8sRUFBRTtNQUNQQyxNQUFNLEVBQUUsUUFBUTtNQUNoQkMsWUFBWSxFQUFFO0lBQ2hCO0VBQ0YsQ0FBQztFQUVELE1BQU1DLG1CQUFtQixHQUFHO0lBQzFCcEIsUUFBUTtJQUNSTixVQUFVO0lBQ1ZLLE1BQU07SUFDTkYsUUFBUTtJQUNSSixZQUFZO0lBQ1pHLE1BQU07SUFDTkU7RUFDRixDQUFDO0VBRURLLE1BQU0sQ0FBQ2tCLEdBQUcsQ0FDUmYsYUFBYSxFQUNiZ0IsWUFBWSxDQUFDRixtQkFBbUIsQ0FDbEMsQ0FBQzs7RUFFRDtFQUNBakIsTUFBTSxDQUFDb0IsSUFBSSxDQUNUakIsYUFBYSxFQUNiZ0IsWUFBWSxDQUFDRixtQkFBbUIsQ0FDbEMsQ0FBQzs7RUFHRDtFQUNBLE1BQU1JLGdCQUFnQixHQUFHLGVBQWU7RUFFeENyQixNQUFNLENBQUNrQixHQUFHLENBQ1I7SUFDRSxHQUFHZixhQUFhO0lBQ2hCQyxJQUFJLEVBQUUsR0FBR2hCLG1CQUFRLEdBQUdpQyxnQkFBZ0I7RUFDdEMsQ0FBQyxFQUNERixZQUFZLENBQUNGLG1CQUFtQixDQUNsQyxDQUFDO0VBRURqQixNQUFNLENBQUNvQixJQUFJLENBQ1Q7SUFDRSxHQUFHakIsYUFBYTtJQUNoQkMsSUFBSSxFQUFFLEdBQUdoQixtQkFBUSxHQUFHaUMsZ0JBQWdCO0VBQ3RDLENBQUMsRUFDREYsWUFBWSxDQUFDRixtQkFBbUIsQ0FDbEMsQ0FBQztBQUNILENBQUMsQ0FBQzs7QUFFSyxTQUFTRSxZQUFZQSxDQUFDO0VBQUV0QixRQUFRO0VBQUVELE1BQU07RUFBRU4sWUFBWTtFQUFFRyxNQUFNO0VBQUVFLGtCQUFrQjtFQUFFSjtBQUFXLENBQUMsRUFBRTtFQUN2RyxPQUFPLGdCQUFnQitCLE9BQU8sRUFBRUMsT0FBTyxFQUFFQyxRQUFRLEVBQUU7SUFDakQsTUFBTUMsUUFBUSxHQUFHRixPQUFPLENBQUNHLEdBQUcsQ0FBQ0MsWUFBWSxDQUFDVCxHQUFHLENBQUMsTUFBTSxDQUFDO0lBRXJELElBQUksQ0FBQ08sUUFBUSxFQUFFO01BQ2IsT0FBT0csaUJBQWlCLENBQUM7UUFDdkJMLE9BQU87UUFDUEMsUUFBUTtRQUNSM0IsUUFBUTtRQUNSRCxNQUFNO1FBQ05ELGtCQUFrQjtRQUNsQmtDLHFCQUFxQixFQUFFdkMsWUFBWSxDQUFDdUMscUJBQXFCO1FBQ3pEcEM7TUFDRixDQUFDLENBQUM7SUFDSjs7SUFFQTtJQUNBLElBQUk7TUFDRjtNQUNBO01BQ0EsTUFBTXFDLGFBQWEsR0FDakIsQ0FBQyxNQUFNeEMsWUFBWSxDQUFDdUMscUJBQXFCLENBQUNFLFFBQVEsQ0FBQ1IsT0FBTyxDQUFDLENBQUNMLEdBQUcsQ0FBQyxDQUFDLEtBQUssQ0FBQyxDQUFDO01BRTFFLE1BQU1jLFVBQVUsR0FBR0YsYUFBYSxDQUFDRyxJQUFJO01BRXJDLElBQUksQ0FBQ0QsVUFBVSxFQUFFO1FBQ2Y7UUFDQTtRQUNBO1FBQ0E7UUFDQSxNQUFNLElBQUlFLEtBQUssQ0FDYiw0RUFDRixDQUFDO01BQ0g7O01BRUE7TUFDQSxPQUFPSixhQUFhLENBQUNHLElBQUk7TUFDekIsTUFBTTNDLFlBQVksQ0FBQ3VDLHFCQUFxQixDQUFDRSxRQUFRLENBQUNSLE9BQU8sQ0FBQyxDQUFDWSxHQUFHLENBQUNMLGFBQWEsQ0FBQztNQUU3RSxNQUFNTSxXQUFXLEdBQUc7UUFDbEJDLElBQUksRUFBRSxNQUFNO1FBQ1pDLFVBQVUsRUFBRWYsT0FBTyxDQUFDRyxHQUFHLENBQUNhLElBQUk7UUFDNUJDLFdBQVcsRUFBRVIsVUFBVSxDQUFDUyxVQUFVO1FBQ2xDQyxFQUFFLEVBQUVWLFVBQVUsQ0FBQ1c7TUFDakIsQ0FBQzs7TUFFRDtNQUNBLE1BQU1DLFlBQVksR0FBRyxNQUFNdEQsWUFBWSxDQUFDdUQsa0JBQWtCLENBQUN0QixPQUFPLEVBQUVhLFdBQVcsQ0FBQztNQUVoRixJQUFJVSxVQUFVLEdBQUcsR0FBRztNQUNwQixJQUFJRixZQUFZLENBQUNHLFdBQVcsRUFBRTtRQUM1QkQsVUFBVSxHQUFHLElBQUFFLGtDQUFlLEVBQUNKLFlBQVksQ0FBQ0csV0FBVyxFQUFFbEQsUUFBUSxDQUFDO01BQ2xFOztNQUVBO01BQ0EsT0FBTzJCLFFBQVEsQ0FBQ3lCLFVBQVUsQ0FBQztRQUN6QkMsT0FBTyxFQUFFO1VBQ1BDLFFBQVEsRUFBRUw7UUFDWjtNQUNGLENBQUMsQ0FBQztJQUNKLENBQUMsQ0FBQyxPQUFPTSxLQUFLLEVBQUU7TUFBQSxJQUFBQyxXQUFBLEVBQUFDLGdCQUFBLEVBQUFDLFlBQUEsRUFBQUMsaUJBQUE7TUFDZC9ELE1BQU0sQ0FBQzJELEtBQUssQ0FBQyxvQ0FBb0MsRUFBRUEsS0FBSyxDQUFDSyxJQUFJLEdBQUdDLElBQUksQ0FBQ0MsU0FBUyxDQUFDUCxLQUFLLENBQUNLLElBQUksQ0FBQyxHQUFHTCxLQUFLLENBQUM7TUFFdEcsSUFBSUYsT0FBTyxHQUFHO1FBQ2ZDLFFBQVEsRUFBRXRELFFBQVEsR0FBRztNQUNwQixDQUFDO01BRUUsSUFBSStELE9BQU8sR0FBRyxFQUFFO01BRWhCLEtBQUFQLFdBQUEsR0FBSUQsS0FBSyxDQUFDSyxJQUFJLGNBQUFKLFdBQUEsZ0JBQUFDLGdCQUFBLEdBQVZELFdBQUEsQ0FBWVEsSUFBSSxjQUFBUCxnQkFBQSxlQUFoQkEsZ0JBQUEsQ0FBa0JGLEtBQUssRUFBRTtRQUFBLElBQUFVLFlBQUEsRUFBQUMsaUJBQUE7UUFDN0JILE9BQU8sQ0FBQ0ksSUFBSSxDQUFDLFNBQVMsR0FBR0Msa0JBQWtCLEVBQUFILFlBQUEsR0FBQ1YsS0FBSyxDQUFDSyxJQUFJLGNBQUFLLFlBQUEsd0JBQUFDLGlCQUFBLEdBQVZELFlBQUEsQ0FBWUQsSUFBSSxjQUFBRSxpQkFBQSx1QkFBaEJBLGlCQUFBLENBQWtCWCxLQUFLLENBQUMsR0FBRyxVQUFVLENBQUM7TUFDckYsQ0FBQyxNQUFNLElBQUlBLEtBQUssQ0FBQ2MsT0FBTyxFQUFFO1FBQ3ZCTixPQUFPLENBQUNJLElBQUksQ0FBQyxTQUFTLEdBQUdDLGtCQUFrQixDQUFDYixLQUFLLENBQUNjLE9BQU8sQ0FBQyxHQUFHLFVBQVUsQ0FBQztNQUMzRTtNQUVHLEtBQUFYLFlBQUEsR0FBSUgsS0FBSyxDQUFDSyxJQUFJLGNBQUFGLFlBQUEsZ0JBQUFDLGlCQUFBLEdBQVZELFlBQUEsQ0FBWU0sSUFBSSxjQUFBTCxpQkFBQSxlQUFoQkEsaUJBQUEsQ0FBa0JXLEtBQUssRUFBRTtRQUFBLElBQUFDLFlBQUEsRUFBQUMsaUJBQUE7UUFDN0JULE9BQU8sQ0FBQ0ksSUFBSSxDQUFDLFNBQVMsR0FBR0Msa0JBQWtCLENBQUNQLElBQUksQ0FBQ0MsU0FBUyxFQUFBUyxZQUFBLEdBQUNoQixLQUFLLENBQUNLLElBQUksY0FBQVcsWUFBQSx3QkFBQUMsaUJBQUEsR0FBVkQsWUFBQSxDQUFZUCxJQUFJLGNBQUFRLGlCQUFBLHVCQUFoQkEsaUJBQUEsQ0FBa0JGLEtBQUssQ0FBQyxDQUFDLEdBQUcsVUFBVSxDQUFDO01BQ3JHO01BRUEsSUFBSVAsT0FBTyxDQUFDVSxNQUFNLEdBQUcsQ0FBQyxFQUFFO1FBQ25CcEIsT0FBTyxDQUFDLFlBQVksQ0FBQyxHQUFHVSxPQUFPO01BQ2pDO01BRUEsT0FBT3BDLFFBQVEsQ0FBQ3lCLFVBQVUsQ0FBQztRQUN6QkM7TUFDRixDQUFDLENBQUM7SUFDSjtFQUNGLENBQUM7QUFDSDs7QUFFQTtBQUNBO0FBQ0E7QUFDQTtBQUNBO0FBQ0E7QUFDQTtBQUNBO0FBQ0E7QUFDQTtBQUNBO0FBQ0EsZUFBZXRCLGlCQUFpQkEsQ0FBQztFQUMvQkwsT0FBTztFQUNQQyxRQUFRO0VBQ1IzQixRQUFRO0VBQ1JELE1BQU07RUFDTkQsa0JBQWtCO0VBQ2xCa0MscUJBQXFCO0VBQ3JCcEM7QUFDRixDQUFDLEVBQUU7RUFDRDtFQUNBO0VBQ0E7RUFDQTtFQUNBLElBQUlnQixPQUFPLEdBQUcsSUFBSTtFQUNsQixJQUFJO0lBQ0YsSUFBSWMsT0FBTyxDQUFDRyxHQUFHLENBQUNDLFlBQVksQ0FBQ1QsR0FBRyxDQUFDLFNBQVMsQ0FBQyxJQUFJcUQsa0JBQWtCLENBQUNoRCxPQUFPLENBQUNHLEdBQUcsQ0FBQ0MsWUFBWSxDQUFDVCxHQUFHLENBQUMsU0FBUyxDQUFDLENBQUMsS0FBSyxHQUFHLEVBQUU7TUFDbEg7TUFDQTtNQUNBVCxPQUFPLEdBQUcsSUFBQXVDLGtDQUFlLEVBQUN6QixPQUFPLENBQUNHLEdBQUcsQ0FBQ0MsWUFBWSxDQUFDVCxHQUFHLENBQUMsU0FBUyxDQUFDLENBQUM7SUFDcEU7RUFDRixDQUFDLENBQUMsT0FBT2tDLEtBQUssRUFBRTtJQUNkO0lBQ0E7RUFBQTtFQUdGLElBQUlvQixVQUFVO0VBQ2QsTUFBTTFDLGFBQWEsR0FBRyxDQUFDLE1BQU1ELHFCQUFxQixDQUFDRSxRQUFRLENBQUNSLE9BQU8sQ0FBQyxDQUFDTCxHQUFHLENBQUMsQ0FBQyxLQUFLLENBQUMsQ0FBQztFQUNqRjtFQUNBO0VBQ0E7RUFDQTtFQUNBOztFQUVBLE1BQU11RCxtQkFBbUIsR0FBR2xELE9BQU8sQ0FBQ0csR0FBRyxDQUFDQyxZQUFZLENBQUNULEdBQUcsQ0FBQyxZQUFZLENBQUMsSUFBSVksYUFBYSxDQUFDYSxVQUFVO0VBQ2xHO0VBQ0E7RUFDQSxPQUFPYixhQUFhLENBQUNhLFVBQVU7RUFDL0IsT0FBT2IsYUFBYSxDQUFDNEMsUUFBUTtFQUU3QixNQUFNQyxnQkFBZ0IsR0FBR0YsbUJBQW1CLEdBQ3ZDN0UsTUFBTSxJQUFLO0lBQ1YsT0FBT0EsTUFBTSxDQUFDOEMsRUFBRSxLQUFLK0IsbUJBQW1CO0VBQzFDLENBQUMsR0FDQTdFLE1BQU0sSUFBSztJQUNWLE9BQU9BLE1BQU0sQ0FBQ2dGLE1BQU0sS0FBSyxNQUFNO0VBQ2pDLENBQUM7RUFFTCxJQUFJO0lBQ0ZKLFVBQVUsR0FBRyxDQUNYLE1BQU03RSxrQkFBa0IsQ0FBQ2tGLGFBQWEsQ0FBQ3BFLE9BQU8sQ0FBQyxFQUMvQ3FFLFlBQVksQ0FBQ0MsSUFBSSxDQUFDSixnQkFBZ0IsQ0FBQztJQUVyQyxJQUFJLENBQUNILFVBQVUsRUFBRTtNQUNmLE1BQU0sSUFBSXRDLEtBQUssQ0FBQyx1QkFBdUIsQ0FBQztJQUMxQztFQUNGLENBQUMsQ0FBQyxPQUFPa0IsS0FBSyxFQUFFO0lBQUEsSUFBQTRCLFlBQUEsRUFBQUMsaUJBQUEsRUFBQUMsWUFBQSxFQUFBQyxpQkFBQTtJQUNkMUYsTUFBTSxDQUFDMkQsS0FBSyxDQUFDLDZEQUE2REEsS0FBSyxDQUFDZ0MsS0FBSyxFQUFFLENBQUM7SUFFM0YsSUFBSWxDLE9BQU8sR0FBRztNQUNiQyxRQUFRLEVBQUV0RCxRQUFRLEdBQUc7SUFDdEIsQ0FBQztJQUVFLElBQUkrRCxPQUFPLEdBQUcsRUFBRTtJQUVoQixLQUFBb0IsWUFBQSxHQUFJNUIsS0FBSyxDQUFDSyxJQUFJLGNBQUF1QixZQUFBLGdCQUFBQyxpQkFBQSxHQUFWRCxZQUFBLENBQVluQixJQUFJLGNBQUFvQixpQkFBQSxlQUFoQkEsaUJBQUEsQ0FBa0I3QixLQUFLLEVBQUU7TUFBQSxJQUFBaUMsWUFBQSxFQUFBQyxpQkFBQTtNQUM3QjFCLE9BQU8sQ0FBQ0ksSUFBSSxDQUFDLFNBQVMsR0FBR0Msa0JBQWtCLEVBQUFvQixZQUFBLEdBQUNqQyxLQUFLLENBQUNLLElBQUksY0FBQTRCLFlBQUEsd0JBQUFDLGlCQUFBLEdBQVZELFlBQUEsQ0FBWXhCLElBQUksY0FBQXlCLGlCQUFBLHVCQUFoQkEsaUJBQUEsQ0FBa0JsQyxLQUFLLENBQUMsR0FBRyxVQUFVLENBQUM7SUFDckYsQ0FBQyxNQUFNLElBQUlBLEtBQUssQ0FBQ2MsT0FBTyxFQUFFO01BQ3ZCTixPQUFPLENBQUNJLElBQUksQ0FBQyxTQUFTLEdBQUdDLGtCQUFrQixDQUFDYixLQUFLLENBQUNjLE9BQU8sQ0FBQyxHQUFHLFVBQVUsQ0FBQztJQUMzRTtJQUVHLEtBQUFnQixZQUFBLEdBQUk5QixLQUFLLENBQUNLLElBQUksY0FBQXlCLFlBQUEsZ0JBQUFDLGlCQUFBLEdBQVZELFlBQUEsQ0FBWXJCLElBQUksY0FBQXNCLGlCQUFBLGVBQWhCQSxpQkFBQSxDQUFrQmhCLEtBQUssRUFBRTtNQUFBLElBQUFvQixZQUFBLEVBQUFDLGlCQUFBO01BQzdCNUIsT0FBTyxDQUFDSSxJQUFJLENBQUMsU0FBUyxHQUFHQyxrQkFBa0IsQ0FBQ1AsSUFBSSxDQUFDQyxTQUFTLEVBQUE0QixZQUFBLEdBQUNuQyxLQUFLLENBQUNLLElBQUksY0FBQThCLFlBQUEsd0JBQUFDLGlCQUFBLEdBQVZELFlBQUEsQ0FBWTFCLElBQUksY0FBQTJCLGlCQUFBLHVCQUFoQkEsaUJBQUEsQ0FBa0JyQixLQUFLLENBQUMsQ0FBQyxHQUFHLFVBQVUsQ0FBQztJQUNyRztJQUVBLElBQUlQLE9BQU8sQ0FBQ1UsTUFBTSxHQUFHLENBQUMsRUFBRTtNQUNuQnBCLE9BQU8sQ0FBQyxZQUFZLENBQUMsR0FBR1UsT0FBTztJQUNqQztJQUVBLE9BQU9wQyxRQUFRLENBQUN5QixVQUFVLENBQUM7TUFDdkJDO0lBQ0osQ0FBQyxDQUFDO0VBQ0o7RUFFQXBCLGFBQWEsQ0FBQ0csSUFBSSxHQUFHO0lBQUVRLFVBQVUsRUFBRStCLFVBQVUsQ0FBQ2hDLFdBQVc7SUFBRUcsVUFBVSxFQUFFNkIsVUFBVSxDQUFDOUIsRUFBRSxJQUFJLElBQUk7SUFBRXBDLEtBQUssRUFBRSxDQUFDO0VBQUUsQ0FBQztFQUN6RyxNQUFNdUIscUJBQXFCLENBQUNFLFFBQVEsQ0FBQ1IsT0FBTyxDQUFDLENBQUNZLEdBQUcsQ0FBQ0wsYUFBYSxDQUFDO0VBRWhFLE9BQU9OLFFBQVEsQ0FBQ3lCLFVBQVUsQ0FBQztJQUN6QkMsT0FBTyxFQUFFO01BQ1BDLFFBQVEsRUFBRXFCLFVBQVUsQ0FBQ2lCO0lBQ3ZCO0VBQ0YsQ0FBQyxDQUFDO0FBQ0oiLCJpZ25vcmVMaXN0IjpbXX0=