"use strict";

Object.defineProperty(exports, "__esModule", {
  value: true
});
exports.SAML_ROUTES = void 0;
exports.defineRoutes = defineRoutes;
var _sanitize_next_url = require("../../sanitize_next_url");
var _configSchema = require("@kbn/config-schema");
var _constants = require("../../../../../utils/constants");
/* eslint-disable @kbn/eslint/require-license-header */
/**
 *    Copyright 2018 floragunn GmbH

 Licensed under the Apache License, Version 2.0 (the "License");
 you may not use this file except in compliance with the License.
 You may obtain a copy of the License at

 http://www.apache.org/licenses/LICENSE-2.0

 Unless required by applicable law or agreed to in writing, software
 distributed under the License is distributed on an "AS IS" BASIS,
 WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 See the License for the specific language governing permissions and
 limitations under the License.
 */

const SAML_ROUTES = exports.SAML_ROUTES = {
  LOGIN: `${_constants.APP_ROOT}/auth/saml/login`
};
function defineRoutes({
  authInstance,
  searchGuardBackend,
  kibanaCore,
  debugLog,
  sessionStorageFactory,
  logger,
  configService
}) {
  const basePath = kibanaCore.http.basePath.serverBasePath;
  const httpResources = kibanaCore.http.resources;
  const router = kibanaCore.http.createRouter();
  const routesPath = '/auth/saml/';

  /**
   * The login page.
   */
  router.get({
    path: `${_constants.APP_ROOT}${routesPath}login`,
    validate: false,
    options: {
      access: 'public',
      authRequired: false
    }
  }, async function (context, request, response) {
    const sessionCookie = (await sessionStorageFactory.asScoped(request).get()) || {};
    /*
    When successful logout,
      headers: {
        cookie: 'searchguard_authentication='
      },
    */

    // Add the nextUrl to the redirect_uri as a parameter. The IDP uses the redirect_uri to redirect the user after successful authentication.
    // For example, it is used to redirect user to the correct dashboard if the user put shared URL in the browser address input before authentication.
    // To make this work, append the wildcard (*) to the valid redirect URI in the IDP configuration, for example
    // https://kibana.example.com:5601/auth/oidc/login*
    let nextUrl = null;
    try {
      if (request.url.searchParams.get('nextUrl') && decodeURIComponent(request.url.searchParams.get('nextUrl')) !== '/') {
        // Do not add the nextUrl to the redirect_uri to avoid the following breaking
        // change for the users that use normal URL to authenticate.
        nextUrl = (0, _sanitize_next_url.sanitizeNextUrl)(request.url.searchParams.get('nextUrl'));
      }
    } catch (error) {
      // We may have received a malformed URL, caught by decodedURIComponent.
      // In this case we just proceed without a nextUrl.
    }
    try {
      // We may have multiple SAML configurations.
      // The authTypeId may still be in the cookie. This happens when
      // a session token expires and no explicit logout is made. We need
      // this behaviour so that we can "refresh" the credentials from the IdP.
      const requestedAuthTypeId = request.url.searchParams.get('authTypeId') || sessionCookie.authTypeId;
      // Delete this again, otherwise the user won't get back to the login page
      // if trying to access Kibana again
      delete sessionCookie.authTypeId;
      delete sessionCookie.authType;
      const authConfigFinder = requestedAuthTypeId ? config => {
        return config.id === requestedAuthTypeId;
      } : config => {
        return config.method === 'saml';
      };
      const authConfig = (await searchGuardBackend.getAuthConfig(nextUrl)).auth_methods.find(authConfigFinder);
      if (!authConfig) {
        throw new Error('Auth config not found');
      }

      // When logging in, sessionCookie={}
      sessionCookie['temp-saml'] = {
        sso_context: authConfig.sso_context,
        authTypeId: authConfig.id || null
      };
      await sessionStorageFactory.asScoped(request).set(sessionCookie);
      return response.redirected({
        headers: {
          location: authConfig.sso_location
        }
      });
    } catch (error) {
      var _error$meta, _error$meta$body, _error$meta3, _error$meta3$body;
      logger.error(`SAML auth, fail to obtain the SAML header: ${error.stack}`);
      var headers = {
        location: basePath + '/searchguard/login?err=saml_init'
      };
      var cookies = [];
      if ((_error$meta = error.meta) !== null && _error$meta !== void 0 && (_error$meta$body = _error$meta.body) !== null && _error$meta$body !== void 0 && _error$meta$body.error) {
        var _error$meta2, _error$meta2$body;
        cookies.push('sg_err=' + encodeURIComponent((_error$meta2 = error.meta) === null || _error$meta2 === void 0 ? void 0 : (_error$meta2$body = _error$meta2.body) === null || _error$meta2$body === void 0 ? void 0 : _error$meta2$body.error) + "; Path=/");
      } else if (error.message) {
        cookies.push('sg_err=' + encodeURIComponent(error.message) + "; Path=/");
      }
      if ((_error$meta3 = error.meta) !== null && _error$meta3 !== void 0 && (_error$meta3$body = _error$meta3.body) !== null && _error$meta3$body !== void 0 && _error$meta3$body.debug) {
        var _error$meta4, _error$meta4$body;
        cookies.push('sg_dbg=' + encodeURIComponent(JSON.stringify((_error$meta4 = error.meta) === null || _error$meta4 === void 0 ? void 0 : (_error$meta4$body = _error$meta4.body) === null || _error$meta4$body === void 0 ? void 0 : _error$meta4$body.debug)) + "; Path=/");
      }
      if (cookies.length > 0) {
        headers['set-cookie'] = cookies;
      }
      return response.redirected({
        headers
      });
    }
  });

  /*
  The page that the IDP redirects to after a successful SP-initiated login.
  */
  router.post({
    path: `${_constants.APP_ROOT}/searchguard/saml/acs`,
    options: {
      access: 'public',
      authRequired: false,
      xsrfRequired: false
    },
    validate: {
      body: _configSchema.schema.object({
        SAMLResponse: _configSchema.schema.string(),
        RelayState: _configSchema.schema.maybe(_configSchema.schema.string())
      }, {
        unknowns: 'allow'
      })
    }
  }, async (context, request, response) => {
    try {
      const {
        body: {
          SAMLResponse,
          RelayState
        } = {}
      } = request;
      /*
      When logging in,
      SAMLResponse = PHNhbWxwOlJlc3BvbnNlIHhtbG5zOnNhbWxwPSJ1cm46b2Fza...
      */

      const sessionCookie = await sessionStorageFactory.asScoped(request).get();
      if (!sessionCookie) {
        throw new Error('The session cookie is absent.');
      }
      /*
        When logging in,
        sessionCookie = {
          "temp-saml": {
            requestId: 'ONELOGIN_a0503578-1ae5-4621-80d5-49336f6d8673',
          }
        }
      */

      const {
        'temp-saml': storedRequestInfo,
        ...restSessionCookie
      } = sessionCookie;
      await sessionStorageFactory.asScoped(request).set(restSessionCookie);
      await authInstance.handleAuthenticate(request, {
        mode: 'saml',
        saml_response: SAMLResponse,
        sso_context: storedRequestInfo.sso_context,
        id: storedRequestInfo.authTypeId
      });
      const nextUrl = RelayState;
      if (nextUrl) {
        return response.redirected({
          headers: {
            location: encodeURI((0, _sanitize_next_url.sanitizeNextUrl)(nextUrl, basePath))
          }
        });
      }
      return response.redirected({
        headers: {
          location: `${basePath}/app/home`
        }
      });
    } catch (error) {
      var _error$meta5, _error$meta5$body, _error$meta7, _error$meta7$body;
      logger.error(`SAML auth, failed to authorize: ${error.stack}`);
      var headers = {
        location: basePath + '/searchguard/login?err=saml'
      };
      var cookies = [];
      if ((_error$meta5 = error.meta) !== null && _error$meta5 !== void 0 && (_error$meta5$body = _error$meta5.body) !== null && _error$meta5$body !== void 0 && _error$meta5$body.error) {
        var _error$meta6, _error$meta6$body;
        cookies.push('sg_err=' + encodeURIComponent((_error$meta6 = error.meta) === null || _error$meta6 === void 0 ? void 0 : (_error$meta6$body = _error$meta6.body) === null || _error$meta6$body === void 0 ? void 0 : _error$meta6$body.error) + "; Path=/");
      } else if (error.message) {
        cookies.push('sg_err=' + encodeURIComponent(error.message) + "; Path=/");
      }
      if ((_error$meta7 = error.meta) !== null && _error$meta7 !== void 0 && (_error$meta7$body = _error$meta7.body) !== null && _error$meta7$body !== void 0 && _error$meta7$body.debug) {
        var _error$meta8, _error$meta8$body;
        cookies.push('sg_dbg=' + encodeURIComponent(JSON.stringify((_error$meta8 = error.meta) === null || _error$meta8 === void 0 ? void 0 : (_error$meta8$body = _error$meta8.body) === null || _error$meta8$body === void 0 ? void 0 : _error$meta8$body.debug)) + "; Path=/");
      }
      if (cookies.length > 0) {
        headers['set-cookie'] = cookies;
      }
      return response.redirected({
        headers
      });
    }
  });

  /**
   * The page that the IdP redirects to after a successful IdP-initiated login
   */
  router.post({
    path: `${_constants.APP_ROOT}/searchguard/saml/acs/idpinitiated`,
    options: {
      access: 'public',
      authRequired: false,
      xsrfRequired: false
    },
    validate: {
      body: _configSchema.schema.object({
        SAMLResponse: _configSchema.schema.string()
      }, {
        unknowns: 'allow'
      })
    }
  }, async (context, request, response) => {
    try {
      await authInstance.handleAuthenticate(request, {
        mode: 'saml',
        saml_response: request.body.SAMLResponse,
        //sso_context: storedRequestInfo.sso_context,
        id: null
      });
      debugLog('Got SAMLResponse: ' + request.body.SAMLResponse);
      return response.redirected({
        headers: {
          location: `${basePath}/app/kibana`
        }
      });
    } catch (error) {
      var _error$meta9, _error$meta9$body, _error$meta11, _error$meta11$body;
      logger.error(`SAML IDP initiated authorization failed: ${error.stack}`);
      var headers = {
        location: basePath + '/searchguard/login?err=saml_idpinitiated'
      };
      var cookies = [];
      if ((_error$meta9 = error.meta) !== null && _error$meta9 !== void 0 && (_error$meta9$body = _error$meta9.body) !== null && _error$meta9$body !== void 0 && _error$meta9$body.error) {
        var _error$meta10, _error$meta10$body;
        cookies.push('sg_err=' + encodeURIComponent((_error$meta10 = error.meta) === null || _error$meta10 === void 0 ? void 0 : (_error$meta10$body = _error$meta10.body) === null || _error$meta10$body === void 0 ? void 0 : _error$meta10$body.error) + "; Path=/");
      } else if (error.message) {
        cookies.push('sg_err=' + encodeURIComponent(error.message) + "; Path=/");
      }
      if ((_error$meta11 = error.meta) !== null && _error$meta11 !== void 0 && (_error$meta11$body = _error$meta11.body) !== null && _error$meta11$body !== void 0 && _error$meta11$body.debug) {
        var _error$meta12, _error$meta12$body;
        cookies.push('sg_dbg=' + encodeURIComponent(JSON.stringify((_error$meta12 = error.meta) === null || _error$meta12 === void 0 ? void 0 : (_error$meta12$body = _error$meta12.body) === null || _error$meta12$body === void 0 ? void 0 : _error$meta12$body.debug)) + "; Path=/");
      }
      if (cookies.length > 0) {
        headers['set-cookie'] = cookies;
      }
      return response.redirected({
        headers
      });
    }
  });

  /**
   * Redirect to logout page after an IdP redirect.
   * This is called after we log out from Kibana,
   * redirect to the IdP and then the IdP redirects
   * back to Kibana.
   */
  const logoutPath = `${_constants.APP_ROOT}/searchguard/saml/logout`;
  const logoutOptions = {
    access: 'public',
    authRequired: false,
    xsrfRequired: false
  };
  const logoutHandler = async (context, request, response) => {
    await authInstance.clear(request);
    return response.redirected({
      headers: {
        location: `${basePath}/`
      }
    });
  };
  // Logout route accepts both GET and POST
  router.get({
    path: logoutPath,
    options: logoutOptions,
    validate: false
  }, logoutHandler);
  router.post({
    path: logoutPath,
    options: logoutOptions,
    validate: false
  }, logoutHandler);
} //end module
//# sourceMappingURL=data:application/json;charset=utf-8;base64,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