"use strict";

var _interopRequireDefault = require("@babel/runtime/helpers/interopRequireDefault");
Object.defineProperty(exports, "__esModule", {
  value: true
});
exports.AuthManager = exports.AUTH_TYPE_NAMES = void 0;
var _defineProperty2 = _interopRequireDefault(require("@babel/runtime/helpers/defineProperty"));
var _coreHttpServer = require("@kbn/core-http-server");
var _lodash = require("lodash");
var _path = _interopRequireDefault(require("path"));
var _coreHttpRouterServerInternal = require("@kbn/core-http-router-server-internal");
/*
 *    Copyright 2021 floragunn GmbH
 *
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 * http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 */

const AUTH_TYPE_NAMES = exports.AUTH_TYPE_NAMES = {
  BASIC: 'basicauth',
  OIDC: 'oidc',
  JWT: 'jwt',
  SAML: 'saml'
};
class AuthManager {
  constructor({
    kibanaCore,
    sessionStorageFactory,
    pluginDependencies,
    logger,
    searchGuardBackend,
    configService,
    spacesService
  }) {
    /**
     * This needs to be the very first onPreAuth handler that
     * we register for the plugin
     * @param request
     * @param response
     * @param toolkit
     * @returns {Promise<*>}
     */
    (0, _defineProperty2.default)(this, "onPreAuth", async (request, response, toolkit) => {
      if (request.headers.authorization) {
        const sessionCookie = (await this.sessionStorageFactory.asScoped(request).get()) || {};
        const authInstance = await this.getAuthInstanceByCookie({
          request
        });
        if (sessionCookie.credentials && authInstance) {
          // In case we already had a session BEFORE we encountered a request
          // with auth headers, we may need to clear the cookie.
          // Make sure to clear any auth related cookie info if we detect a different header
          await authInstance.clear(request, true);
        }
      }
      return toolkit.next();
    });
    (0, _defineProperty2.default)(this, "checkAuth", async (request, response, toolkit) => {
      const sessionCookie = (await this.sessionStorageFactory.asScoped(request).get()) || {};

      /**
       * If we have an auth header, just let the request pass through
       */
      if (request.headers.authorization) {
        return toolkit.next();
      }
      try {
        if (request.route.options.authRequired === false) {
          return toolkit.next();
        }
      } catch (error) {
        this.logger.info('Could not read auth options for the path: ' + request.url.pathname);
      }

      /**
       * This block isn't really needed since we started checking
       * the route's auth options, but leaving it in case there
       * are any edge cases
       */
      if (this.routesToIgnore.includes(request.url.pathname) || request.url.pathname.indexOf('/bundle') > -1) {
        return toolkit.next();
      }

      /**
       * This block isn't really needed since we started checking
       * the route's auth options, but leaving it in case there
       * are any edge cases.
       * These routes are used configurable.
       */
      if (this.unauthenticatedRoutes.includes(request.url.pathname)) {
        // If we do this, we don't really assign any relevant headers
        // Until now, we got the kibana server user here, but those credentials were
        // not really used, it seems

        return toolkit.next();
      }
      const authInstanceByRequest = await this.getAuthInstanceByRequest({
        request
      });
      if (authInstanceByRequest) {
        return authInstanceByRequest.checkAuth(request, response, toolkit);
      }

      // @todo This way of handling anonymous auth unfortunately
      // doesn't provide a good way of showing an error message
      // if the SG backend hasn't been configured properly
      if (!sessionCookie.authType && this.configService.get('searchguard.auth.anonymous_auth_enabled')) {
        /*
        return toolkit.authenticated({
        });
        */

        return toolkit.next();
      }

      // Here comes the not authenticated logic...

      // We don't have any cookie, but we may have an optional auth
      try {
        if (request.route.options.authRequired === 'optional') {
          return toolkit.next();
        }
      } catch (error) {
        this.logger.info('Could not read auth options for the path: ' + request.url.pathname);
      }
      const isAjaxRequest = request.headers && (request.headers.accept && request.headers.accept.split(',').indexOf('application/json') > -1 || request.headers['content-type'] && request.headers['content-type'].indexOf('application/json') > -1);
      const nextUrl = this.getNextUrl(request);
      let loginPageURL = this.basePath + '/searchguard/login' + `?nextUrl=${nextUrl}`;
      try {
        const authConfig = await this.searchGuardBackend.getAuthConfig(nextUrl);
        let config;
        if (authConfig && authConfig.auth_methods && authConfig.auth_methods.length == 1 && authConfig.auth_methods[0].sso_location) {
          // If there is only one auth_method with sso_location
          config = authConfig.auth_methods[0];
        } else {
          // If one of the methods has auto_select property enabled
          config = authConfig && authConfig.auth_methods && authConfig.auth_methods.find(({
            auto_select
          }) => auto_select);
        }
        if (config && config.sso_location) {
          loginPageURL = config.sso_location;
          const authInstance = this.authInstances[config.method];
          if (authInstance && authInstance.loginURL) {
            loginPageURL = new URL(this.basePath + authInstance.loginURL, 'http://abc');
            if (config.id) {
              loginPageURL.searchParams.set('authTypeId', config.id);
            }
            if (nextUrl) {
              loginPageURL.searchParams.set('nextUrl', nextUrl);
            }
            loginPageURL = loginPageURL.href.replace(loginPageURL.origin, '');
          }
          if (config.capture_url_fragment && nextUrl && !isAjaxRequest) {
            return response.redirected({
              headers: {
                'location': `${this.basePath}/auth/captureurlfragment?loginHandler=${this.basePath + authInstance.loginURL}&authTypeId=${config.id}&nextUrl=${encodeURIComponent(nextUrl)}`
              }
            });
          }
        }
      } catch (error) {
        console.error("Error while retrieving auth config", error);
      }
      if (isAjaxRequest) {
        // If the session has expired, we may receive ajax requests that can't handle a 302 redirect.
        // In this case, we trigger a 401 and let the interceptor handle the redirect on the client side.
        return response.unauthorized({
          headers: {
            sg_redirectTo: loginPageURL
          },
          body: {
            message: 'Session expired or invalid username and password'
          }
        });
      }
      return response.redirected({
        headers: {
          location: loginPageURL
        }
      });
    });
    /**
     * Handler for validating the auth state in routes with authRequired === optional.
     *
     * Our auth logic runs in the onPreAuth lifecycle step, and
     * we let requests with optional auth pass through.
     *
     * Subsequently, Kibana's auth logic will consider the
     * request authenticated in the default auth logic.
     *
     * If the optional route's handler behaves differently
     * for authenticated request and makes calls to the
     * backend, we may run into unexpected 401s.
     *
     * This handler tries to sync the authentication
     * status with the rest of our auth logic.
     * If there's no authorization header attached to the
     * request, we will consider it unauthenticated.
     *
     * @see https://github.com/elastic/kibana/blob/6c438b331c703c507af524a13aab634cdbfbbb13/dev_docs/tutorials/endpoints.mdx?plain=1#L399
     *
     * @param request
     * @param response
     * @param toolkit
     * @returns {Promise<*>}
     */
    (0, _defineProperty2.default)(this, "handleAuthForOptionalRoutes", async (request, response, toolkit) => {
      try {
        if (request.route.options.authRequired === 'optional' && !request.headers.authorization) {
          const rawRequest = (0, _coreHttpRouterServerInternal.ensureRawRequest)(request);
          rawRequest.auth.isAuthenticated = false;
        }
      } catch (error) {
        this.logger.info('handleAuthForOptionalRoute could not read auth options for the path: ' + request.url.pathname);
      }
      return toolkit.next();
    });
    /**
     * Handler for validating the auth state in routes with authRequired === optional.
     *
     * Our auth logic runs in the onPreAuth lifecycle step, and
     * we let requests with optional auth pass through.
     *
     * Subsequently, Kibana's auth logic will consider the
     * request authenticated in the default auth logic.
     *
     * If the optional route's handler behaves differently
     * for authenticated request and makes calls to the
     * backend, we may run into unexpected 401s.
     *
     * This handler tries to sync the authentication
     * status with the rest of our auth logic.
     * If there's no authorization header attached to the
     * request, we will consider it unauthenticated.
     *
     * @see https://github.com/elastic/kibana/blob/6c438b331c703c507af524a13aab634cdbfbbb13/dev_docs/tutorials/endpoints.mdx?plain=1#L399
     *
     * @param request
     * @param response
     * @param toolkit
     * @returns {Promise<*>}
     */
    (0, _defineProperty2.default)(this, "handleAuthForOptionalRoutes", async (request, response, toolkit) => {
      try {
        if (request.auth.isAuthenticated === true && request.route.options.authRequired === 'optional' && !request.headers.authorization) {
          // Set isAuthenticated to false
          if (['/login'].indexOf(request.url.pathname) === -1) {
            const rawRequest = (0, _coreHttpRouterServerInternal.ensureRawRequest)(request);
            rawRequest.auth.isAuthenticated = false;
          }
        }
      } catch (error) {
        this.logger.info('handleAuthForOptionalRoute could not read auth options for the path: ' + request.url.pathname);
      }
      return toolkit.next();
    });
    // @todo Not needed for 7.10?
    (0, _defineProperty2.default)(this, "onPostAuth", async (request, response, toolkit) => {
      if (request.route.path === '/api/core/capabilities') {
        if (this.configService.get('searchguard.auth.anonymous_auth_enabled')) return toolkit.next();
        const authHeaders = await this.getAllAuthHeaders(request);
        if (authHeaders === false && !request.headers.authorization) {
          /*
          We need this redirect because Kibana calls the capabilities on our login page. The Kibana checks if there is the default space in the Kibana index.
          The problem is that the Kibana call is scoped to the current request. And the current request doesn't contain any credentials in the headers because the user hasn't been authenticated yet.
          As a result, the call fails with 401, and the user sees the Kibana error page instead of our login page.
          We flank this issue by redirecting the Kibana call to our route /api/v1/searchguard/kibana_capabilities where we serve some
          minimum amount of capabilities. We expect that Kibana fetches the capabilities again once the user logged in.
          */
          // The payload is passed together with the redirect despite of the undefined here
          return new _coreHttpServer.KibanaResponse(307, undefined, {
            headers: {
              location: this.basePath + '/api/v1/searchguard/kibana_capabilities'
            }
          });
        }
      }
      return toolkit.next();
    });
    this.kibanaCore = kibanaCore;
    this.sessionStorageFactory = sessionStorageFactory;
    this.searchGuardBackend = searchGuardBackend;
    this.logger = logger;
    this.pluginDependencies = pluginDependencies;
    this.configService = configService;
    this.spacesService = spacesService;
    this.authInstances = {};
    this.unauthenticatedRoutes = this.configService.get('searchguard.auth.unauthenticated_routes');

    /**
     * We used to have a list of paths that auth should ignore.
     * Now, we check the route options instead - if no auth
     * is required, we just skip the path.
     *
     * Keeping this in case there are exceptions to this rule.
     */
    this.routesToIgnore = ['/internal/security/user_profile'];
    this.basePath = kibanaCore.http.basePath.get();
  }
  registerAuthInstances() {
    [require('./types/openid/OpenId'), require('./types/basicauth/BasicAuth'), require('./types/jwt/Jwt'), require('./types/saml/Saml')].forEach(AuthClass => {
      // @todo This needs to respect the order as given by the backend
      const authInstance = new AuthClass({
        kibanaCore: this.kibanaCore,
        sessionStorageFactory: this.sessionStorageFactory,
        pluginDependencies: this.pluginDependencies,
        logger: this.logger,
        searchGuardBackend: this.searchGuardBackend,
        config: this.configService,
        authManager: this,
        // @todo Is the authManager used?
        spacesService: this.spacesService
      });
      authInstance.init();
      this.authInstances[authInstance.type] = authInstance;
    });
  }
  registerAuthInstance(authTypeName, authInstance) {
    this.authInstances[authTypeName] = authInstance;
  }
  getAuthInstanceByName(authTypeName) {
    if (this.authInstances[authTypeName]) {
      return this.authInstances[authTypeName];
    }
    return null;
  }
  async getAuthInstanceByRequest({
    request
  }) {
    const matchedAuthInstance = await this.getAuthInstanceByAuthTypes({
      request
    });
    // matchedAuthInstance will be null if we didn't get a match
    if (matchedAuthInstance) {
      return matchedAuthInstance;
    }
    const authInstanceByCookie = await this.getAuthInstanceByCookie({
      request
    });
    if (authInstanceByCookie) {
      return authInstanceByCookie;
    }
    return null;
  }
  async getAuthInstanceByAuthTypes({
    request
  }) {
    for (const authType in this.authInstances) {
      const authInstance = this.getAuthInstanceByName(authType);
      const authInstanceResult = await authInstance.detectCredentialsByRequest({
        request
      });
      if (authInstanceResult !== null) {
        return authInstance;
      }
    }
    return null;
  }
  async getAuthInstanceByCookie({
    request
  }) {
    const sessionCookie = (await this.sessionStorageFactory.asScoped(request).get()) || {};
    if (sessionCookie.authType && this.authInstances[sessionCookie.authType]) {
      return this.getAuthInstanceByName(sessionCookie.authType);
    }
    return null;
  }
  getNextUrl(request) {
    let nextUrl = _path.default.posix.join(this.basePath, request.url.pathname);
    if (request.url.search) nextUrl += request.url.search;
    return nextUrl;
  }

  /**
   * Get credentials from an existing cookie only
   * @param request
   * @returns {Promise<*|boolean|boolean>}
   */
  async getAuthHeader(request) {
    const authInstance = await this.getAuthInstanceByCookie({
      request
    });
    if (authInstance) {
      // @todo A bit weird that we have different method signatures here
      const sessionCookie = (await this.sessionStorageFactory.asScoped(request).get()) || {};
      return authInstance.getAuthHeader(sessionCookie);
    }
    return false;
  }
  async getAllAuthHeaders(request) {
    if (request.headers.authorization) {
      return false;
    }
    const authInstance = await this.getAuthInstanceByRequest({
      request
    });
    if (authInstance) {
      return authInstance.getAllAuthHeaders(request);
    }
    return false;
  }
  async logout({
    context,
    request,
    response
  }) {
    const authInstance = await this.getAuthInstanceByCookie({
      request
    });
    if (authInstance) {
      return await authInstance.logout({
        context,
        request,
        response
      });
    }
    return response.ok();
  }
}
exports.AuthManager = AuthManager;
//# sourceMappingURL=data:application/json;charset=utf-8;base64,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