"use strict";

Object.defineProperty(exports, "__esModule", {
  value: true
});
exports.deleteAuthToken = deleteAuthToken;
exports.getAuthToken = getAuthToken;
exports.getAuthTokenServiceInfo = getAuthTokenServiceInfo;
exports.getAuthTokens = getAuthTokens;
exports.registerRoutes = registerRoutes;
exports.saveAuthToken = saveAuthToken;
var _configSchema = require("@kbn/config-schema");
var _utils = require("../../utils");
var _constants = require("../../utils/constants");
/*
 *    Copyright 2020 floragunn GmbH
 *
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 * http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 */

function getAuthTokens({
  clusterClient,
  fetchAllFromScroll,
  logger
}) {
  return async function (context, request, response) {
    try {
      const {
        body: {
          scroll,
          ...bodyProps
        }
      } = request;
      let allHits = [];
      if (scroll) {
        const firstScrollResponse = await clusterClient.asScoped(request).asCurrentUser.transport.request({
          method: 'post',
          path: `/_searchguard/authtoken/_search?scroll=${scroll}`,
          body: bodyProps
        });
        allHits = await fetchAllFromScroll({
          clusterClient,
          scroll,
          request,
          response: firstScrollResponse
        });
      } else {
        const body = await clusterClient.asScoped(request).asCurrentUser.transport.request({
          method: 'post',
          path: '/_searchguard/authtoken/_search',
          body: bodyProps
        });
        allHits = body.hits.hits;
      }
      return response.ok({
        body: allHits.map(({
          _source,
          _id
        }) => ({
          ..._source,
          _id
        }))
      });
    } catch (err) {
      logger.error(`getAuthTokens: ${err.stack}`);
      return response.customError((0, _utils.wrapForCustomError)(err));
    }
  };
}
function deleteAuthToken({
  clusterClient,
  logger
}) {
  return async function (context, request, response) {
    try {
      const {
        params: {
          id
        }
      } = request;
      const body = await clusterClient.asScoped(request).asCurrentUser.transport.request({
        method: 'delete',
        path: `/_searchguard/authtoken/${id}`
      });
      return response.ok({
        body
      });
    } catch (err) {
      logger.error(`deleteAuthToken: ${err.stack}`);
      return response.customError((0, _utils.wrapForCustomError)(err));
    }
  };
}
function getAuthToken({
  clusterClient,
  logger
}) {
  return async function (context, request, response) {
    try {
      const {
        params: {
          id
        }
      } = request;
      const body = await clusterClient.asScoped(request).asCurrentUser.transport.request({
        method: 'get',
        path: `/_searchguard/authtoken/${id}`
      });
      return response.ok({
        body
      });
    } catch (err) {
      logger.error(`getAuthToken: ${err.stack}`);
      return response.customError((0, _utils.wrapForCustomError)(err));
    }
  };
}
function saveAuthToken({
  clusterClient,
  logger
}) {
  return async function (context, request, response) {
    try {
      const {
        body: authToken
      } = request;
      const body = await clusterClient.asScoped(request).asCurrentUser.transport.request({
        method: 'post',
        path: '/_searchguard/authtoken',
        body: authToken
      });
      return response.ok({
        body
      });
    } catch (err) {
      logger.error(`saveAuthToken: ${err.stack}`);
      return response.customError((0, _utils.wrapForCustomError)(err));
    }
  };
}
function getAuthTokenServiceInfo({
  clusterClient,
  logger
}) {
  return async function (context, request, response) {
    try {
      const body = await clusterClient.asScoped(request).asCurrentUser.transport.request({
        method: 'get',
        path: `/_searchguard/authtoken/_info`
      });
      return response.ok({
        body
      });
    } catch (err) {
      logger.error(`getAuthTokenServiceInfo: ${err.stack}`);
      return response.customError((0, _utils.wrapForCustomError)(err));
    }
  };
}
function registerRoutes({
  router,
  clusterClient,
  logger
}) {
  router.post({
    path: `${_constants.API_ROOT}/searchguard_authtokens/authtoken/_search`,
    validate: {
      body: _configSchema.schema.object({
        scroll: _configSchema.schema.maybe(_configSchema.schema.string()),
        query: _configSchema.schema.maybe(_configSchema.schema.object({}, {
          unknowns: 'allow'
        }))
      }, {
        unknowns: 'allow'
      })
    },
    options: {
      authRequired: true
    }
  }, getAuthTokens({
    clusterClient,
    fetchAllFromScroll: _utils.fetchAllFromScroll,
    logger
  }));
  router.delete({
    path: `${_constants.API_ROOT}/searchguard_authtokens/authtoken/{id}`,
    validate: {
      params: _configSchema.schema.object({
        id: _configSchema.schema.string()
      })
    },
    options: {
      authRequired: true
    }
  }, deleteAuthToken({
    clusterClient,
    logger
  }));
  router.get({
    path: `${_constants.API_ROOT}/searchguard_authtokens/authtoken/{id}`,
    validate: {
      params: _configSchema.schema.object({
        id: _configSchema.schema.string()
      })
    },
    options: {
      authRequired: true
    }
  }, getAuthToken({
    clusterClient,
    logger
  }));
  router.post({
    path: `${_constants.API_ROOT}/searchguard_authtokens/authtoken`,
    validate: {
      body: _configSchema.schema.object({}, {
        unknowns: 'allow'
      })
    },
    options: {
      authRequired: true
    }
  }, saveAuthToken({
    clusterClient,
    logger
  }));
  router.get({
    path: `${_constants.API_ROOT}/searchguard_authtokens/authtoken/_info`,
    validate: false,
    options: {
      authRequired: true
    }
  }, getAuthTokenServiceInfo({
    clusterClient,
    logger
  }));
}
//# sourceMappingURL=data:application/json;charset=utf-8;base64,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