"use strict";

var _interopRequireDefault = require("@babel/runtime/helpers/interopRequireDefault");
Object.defineProperty(exports, "__esModule", {
  value: true
});
exports.default = void 0;
var _defineProperty2 = _interopRequireDefault(require("@babel/runtime/helpers/defineProperty"));
var _lodash = require("lodash");
var _coreHttpServer = require("@kbn/core-http-server");
var _coreHttpRouterServerInternal = require("@kbn/core-http-router-server-internal");
var _invalid_session_error = _interopRequireDefault(require("../errors/invalid_session_error"));
var _session_expired_error = _interopRequireDefault(require("../errors/session_expired_error"));
var _filter_auth_headers = _interopRequireDefault(require("../filter_auth_headers"));
var _missing_tenant_error = _interopRequireDefault(require("../errors/missing_tenant_error"));
var _path = _interopRequireDefault(require("path"));
var _multitenancy = require("../../../../../common/multitenancy");
/* eslint-disable @kbn/eslint/require-license-header */
/**
 *    Copyright 2018 floragunn GmbH

 Licensed under the Apache License, Version 2.0 (the "License");
 you may not use this file except in compliance with the License.
 You may obtain a copy of the License at

 http://www.apache.org/licenses/LICENSE-2.0

 Unless required by applicable law or agreed to in writing, software
 distributed under the License is distributed on an "AS IS" BASIS,
 WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 See the License for the specific language governing permissions and
 limitations under the License.
 */

class AuthType {
  constructor({
    searchGuardBackend,
    kibanaCore,
    config,
    logger,
    sessionStorageFactory,
    pluginDependencies,
    spacesService
  }) {
    /**
     * Called internally and checks for an AJAX request before
     * invoking the auth type's OnUnAuthenticated method.
     * @param request
     * @param response
     * @param toolkit
     * @param error
     * @returns {Promise<void|*>}
     * @private
     */
    (0, _defineProperty2.default)(this, "_handleUnAuthenticated", async (request, response, toolkit, error = null) => {
      // We don't have valid cookie credentials, but we may have an optional auth
      try {
        if (request.route.options.authRequired === 'optional') {
          return toolkit.next();
        }
      } catch (error) {
        this.logger.info('Could not read auth options for the path: ' + request.url.pathname);
      }
      if (request.headers) {
        // If the session has expired, we may receive ajax requests that can't handle a 302 redirect.
        // In this case, we trigger a 401 and let the interceptor handle the redirect on the client side.
        if (request.headers.accept && request.headers.accept.split(',').indexOf('application/json') > -1 || request.headers['content-type'] && request.headers['content-type'].indexOf('application/json') > -1) {
          this.debugLog('Not authenticated, detected AJAX request');
          const sessionCookie = (await this.sessionStorageFactory.asScoped(request).get()) || {};
          if (request.route.path === '/api/core/capabilities') {
            return toolkit.notHandled();
          }
          return response.unauthorized({
            headers: {
              sg_redirectTo: await this.getRedirectTargetForUnauthenticated(request, error, true, sessionCookie)
            },
            body: {
              message: 'Session expired'
            }
          });
        }
      }
      return this.onUnAuthenticated(request, response, toolkit, error);
    });
    (0, _defineProperty2.default)(this, "onPostAuth", async (request, response, toolkit) => {
      if (request.route.path === '/api/core/capabilities') {
        const sessionCookie = (await this.sessionStorageFactory.asScoped(request).get()) || {};
        if (sessionCookie.isAnonymousAuth) return toolkit.next();
        const authHeaders = await this.getAllAuthHeaders(request);
        if (authHeaders === false) {
          /*
          We need this redirect because Kibana calls the capabilities on our login page. The Kibana checks if there is the default space in the Kibana index.
          The problem is that the Kibana call is scoped to the current request. And the current request doesn't contain any credentials in the headers because the user hasn't been authenticated yet.
          As a result, the call fails with 401, and the user sees the Kibana error page instead of our login page.
          We flank this issue by redirecting the Kibana call to our route /api/v1/searchguard/kibana_capabilities where we serve some
          minimum amount of capabilities. We expect that Kibana fetches the capabilities again once the user logged in.
          */
          // The payload is passed together with the redirect despite of the undefined here
          return new _coreHttpServer.KibanaResponse(307, undefined, {
            headers: {
              location: this.basePath + '/api/v1/searchguard/kibana_capabilities'
            }
          });
        } else {
          // Update the request with auth headers in order to allow Kibana to check the default space.
          // Kibana page breaks if Kibana can't check the default space.
          const rawRequest = (0, _coreHttpRouterServerInternal.ensureRawRequest)(request);
          (0, _lodash.assign)(rawRequest.headers, authHeaders);
        }
      }
      return toolkit.next();
    });
    (0, _defineProperty2.default)(this, "checkAuth", async (request, response, toolkit) => {
      let sessionCookie = (await this.sessionStorageFactory.asScoped(request).get()) || {};
      try {
        sessionCookie = await this.getCookieWithCredentials(request);
      } catch (error) {
        return this._handleUnAuthenticated(request, response, toolkit, error);
      }
      if (sessionCookie.credentials) {
        const authHeaders = await this.getAllAuthHeaders(request, sessionCookie);
        if (!authHeaders) {
          this.logger.error(`An error occurred while computing auth headers, clearing session: No headers found in the session cookie`);
          await this.clear(request);
          return this._handleUnAuthenticated(request, response, toolkit);
        }
        const isMtEnabled = this.config.get('searchguard.multitenancy.enabled');
        if (!isMtEnabled && this.pluginDependencies.spaces) {
          await this.spacesService.createDefaultSpace({
            request: {
              headers: authHeaders
            }
          });
        }
        const rawRequest = (0, _coreHttpRouterServerInternal.ensureRawRequest)(request);
        (0, _lodash.assign)(rawRequest.headers, authHeaders);
        return toolkit.next();
      }
      return this._handleUnAuthenticated(request, response, toolkit);
    });
    this.searchGuardBackend = searchGuardBackend;
    this.config = config;
    this.kibanaCore = kibanaCore;
    this.logger = logger;
    this.sessionStorageFactory = sessionStorageFactory;
    this.pluginDependencies = pluginDependencies;
    this.spacesService = spacesService;
    this.basePath = kibanaCore.http.basePath.get();
    this.frontendBaseUrl = this.config.get('searchguard.frontend_base_url') || kibanaCore.http.basePath.publicBaseUrl;
    this.sgFrontendConfigId = this.config.get('searchguard.sg_frontend_config_id') || 'default';
    if (!this.frontendBaseUrl) {
      const serverInfo = kibanaCore.http.getServerInfo();
      this.frontendBaseUrl = serverInfo.protocol + '://' + serverInfo.hostname + ':' + serverInfo.port + '/' + kibanaCore.http.basePath.serverBasePath;
    }
    this.authDebugEnabled = this.config.get('searchguard.auth.debug');

    /**
     * The authType is saved in the auth cookie for later reference
     * @type {string}
     */
    this.type = null;

    /**
     * If a loginURL is defined, we can skip the auth selector page
     * if the customer only has one auth type enabled.
     * @type {string|null}
     */
    this.loginURL = null;

    /**
     * Tells the sessionPlugin whether or not to validate the number of tenants when authenticating
     * @type {boolean}
     */
    this.validateAvailableTenants = true;

    /**
     * The name of the header were we look for an authorization value.
     * This should most likely be set in the subclass depending on a config value.
     * @type {string}
     */
    this.authHeaderName = 'authorization';

    /**
     * Additional headers that should be passed as part as the authentication.
     * Do not use headers here that have an effect on which user is logged in.
     * @type {string[]}
     */
    this.allowedAdditionalAuthHeaders = ['sg_impersonate_as'];
  }
  async init() {
    this.setupRoutes();
  }

  /**
   * Returns the auth header needed for the Search Guard backend
   * @param session
   * @returns {*}
   */
  getAuthHeader(session) {
    if (session.credentials && session.credentials.authHeaderValue) {
      return {
        [this.authHeaderName]: session.credentials.authHeaderValue
      };
    }
    return false;
  }

  /**
   * Can be used by auth types that need to handle cases
   * where the credentials are passed together with the
   * request.
   * Example: JWT supports passing the bearer token per query parameter
   *
   * NB: Should NOT be used to detect pre-authenticated requests.
   * For those, we don't want to create a cookie.
   *
   * @param request
   * @returns {Promise<null>}
   */
  async detectCredentialsByRequest({
    request
  }) {
    return null;
  }

  /**
   * Checks if we have an authorization header.
   *
   * Pass the existing session credentials to compare with the authorization header.
   *
   * @param request
   * @param sessionCredentials
   * @returns {object|null} - credentials for the authentication
   */
  detectAuthHeaderCredentials(request, sessionCredentials = null) {
    if (request.headers[this.authHeaderName]) {
      const authHeaderValue = request.headers[this.authHeaderName];

      // If we have sessionCredentials AND auth headers we need to check if they are the same.
      if (sessionCredentials !== null && sessionCredentials.authHeaderValue === authHeaderValue) {
        // The auth header credentials are the same as those in the session,
        // no need to return new credentials so we're just returning null here
        return null;
      }
      return {
        authHeaderValue: authHeaderValue
      };
    }
    return null;
  }
  async getRedirectTargetForUnauthenticated() {
    throw new Error('The getRedirectTargetForUnauthenticated method must be implemented by the sub class');
  }
  async authenticate(credentials, options = {}, additionalAuthHeaders = {}) {
    try {
      this.debugLog('Authenticating using ' + credentials);
      credentials.frontend_base_url = this.frontendBaseUrl;
      credentials.config_id = this.sgFrontendConfigId;
      const sessionResponse = await this.searchGuardBackend.authenticateWithSession(credentials);
      const sessionCredentials = {
        authHeaderValue: 'Bearer ' + sessionResponse.token
      };
      this.debugLog('Token ' + sessionCredentials.authHeaderValue);
      const user = await this.searchGuardBackend.authenticateWithHeader(this.authHeaderName, sessionCredentials.authHeaderValue, additionalAuthHeaders);
      const session = {
        username: user.username,
        // The session token
        credentials: sessionCredentials,
        authType: this.type,
        authTypeId: credentials.id
      };
      return {
        session,
        user,
        redirectUri: sessionResponse.redirect_uri
      };
    } catch (error) {
      throw error;
    }
  }
  async onUnAuthenticated(request, response, toolkit, error = null) {
    const redirectTo = await this.getRedirectTargetForUnauthenticated(request, error);
    return response.redirected({
      headers: {
        location: `${redirectTo}`
      }
    });
  }

  /**
   * A helper for generating the correct nextUrl.
   * Spaces manipulates the URL for non default
   * spaces, and that change is not reflected
   * in request.url.pathname
   * @param request
   * @returns {string}
   */
  getNextUrl(request) {
    return _path.default.posix.join(this.basePath, request.url.pathname) + request.url.search;
  }
  setupRoutes() {
    throw new Error('The getAuthHeader method must be implemented by the sub class');
  }
  async getCookieWithCredentials(request) {
    const authHeaderCredentials = await this.detectCredentialsByRequest({
      request
    });
    if (authHeaderCredentials) {
      try {
        this.debugLog('Got auth header credentials, trying to authenticate');
        const {
          session
        } = await this.handleAuthenticate(request, authHeaderCredentials);
        return session;
      } catch (error) {
        this.logger.error(`Got auth header credentials, but authentication failed: ${error.stack}`);
        // Fall through
      }
    }
    let sessionCookie = (await this.sessionStorageFactory.asScoped(request).get()) || {};
    if (sessionCookie.credentials) {
      try {
        return await this.validateSessionCookie(request, sessionCookie);
      } catch (error) {
        // Logging this as info since it isn't really an error, but just a part of the flow.
        this.logger.info(`Got credentials, but the validation failed: ${error.stack}`);
        throw error;
      }
    }

    // No (valid) cookie, we need to check for headers
    return sessionCookie;
  }
  /**
   * If a session auth cookie exists, the sessionValidator is called to validate the content.
   * If the cookie isn't valid, an error will be thrown
   * @param server
   * @returns {validate}
   */
  async validateSessionCookie(request, sessionCookie) {
    if (sessionCookie.authType !== this.type) {
      await this.clear(request);
      throw new _invalid_session_error.default('Invalid cookie');
    }
    const checkTokenExpirationTime = Date.now();
    if (!sessionCookie.checkTokenExpirationTime || checkTokenExpirationTime - sessionCookie.checkTokenExpirationTime > 15000) {
      try {
        const authHeader = this.getAuthHeader(sessionCookie);
        const authInfoResponse = await this.searchGuardBackend.authinfo(authHeader || request.headers);
        sessionCookie.checkTokenExpirationTime = checkTokenExpirationTime;
        await this.sessionStorageFactory.asScoped(request).set(sessionCookie);
        if (authInfoResponse.user_name !== sessionCookie.username) {
          throw new Error('We have a different user in the cookie. Most likely because of anonymous auth.');
        }
      } catch (error) {
        await this.clear(request);
        throw new _session_expired_error.default('Session expired');
      }
    }
    return sessionCookie;
  }

  /**
   * Get all auth headers based on the current request.
   *
   * @param request
   * @param sessionCookie
   * @returns {Promise<boolean|*>}
   */
  async getAllAuthHeaders(request, sessionCookie = null) {
    if (!sessionCookie) {
      try {
        sessionCookie = await this.getCookieWithCredentials(request);
      } catch (error) {
        this.logger.info(`Getting all auth headers failed: ${error.stack}`);
      }
    }
    if (!sessionCookie || !sessionCookie.credentials) {
      return false;
    }
    const authHeader = this.getAuthHeader(sessionCookie);
    if (authHeader !== false) {
      this.addAdditionalAuthHeaders(request, authHeader, sessionCookie);
      return authHeader;
    }
    return false;
  }

  /**
   * @deprecated
   *
   * Method for adding additional auth type specific authentication headers.
   * Override this in the auth type for type specific headers.
   *
   * NB: Remember to call the super method if you do.
   *
   * @param request
   * @param authHeader
   * @param session
   */
  addAdditionalAuthHeaders(request, authHeader, session) {
    if (session.additionalAuthHeaders) {
      for (const header in session.additionalAuthHeaders) {
        if (session.additionalAuthHeaders.hasOwnProperty(header)) {
          authHeader[header] = session.additionalAuthHeaders[header];
        }
      }
    }
  }
  debugLog(message, label = this.type) {
    if (this.authDebugEnabled) {
      try {
        if (typeof message !== 'string') {
          message = JSON.stringify(message);
        }
        this.logger.info(`${label} ${message}`);
      } catch (error) {
        this.logger.error(`Error in debug log: ${error.stack}`);
      }
    }
  }

  /**
   * Tries to authenticate a user. If multitenancy is enabled, we also try to validate that the
   * user has at least one valid tenant
   * @param {object} request
   * @param {object} headers
   * @param {object} credentials
   * @returns {Promise<*>}
   */
  async handleAuthenticate(request, credentials, options = {}) {
    try {
      const additionalAuthHeaders = (0, _filter_auth_headers.default)(request.headers, this.allowedAdditionalAuthHeaders);
      // authResponse is an object with .session and .user
      const authResponse = await this.authenticate(credentials, options, additionalAuthHeaders);
      return this._handleAuthResponse(request, credentials, authResponse, additionalAuthHeaders);
    } catch (error) {
      // Make sure we clear any existing cookies if something went wrong
      this.clear(request, true);
      throw error;
    }
  }

  /**
   * Normalized response after an authentication
   * @param credentials
   * @param authResponse
   * @returns {*}
   * @private
   */
  async _handleAuthResponse(request, credentials, authResponse, additionalAuthHeaders = {}) {
    // Make sure the user has a tenant that they can use

    if (this.validateAvailableTenants && this.config.get('searchguard.multitenancy.enabled')) {
      let userTenantInfo;
      let allTenants = {};
      try {
        userTenantInfo = await this.searchGuardBackend.getUserTenantInfo({
          authorization: authResponse.session.credentials.authHeaderValue
        });
        userTenantInfo = this.searchGuardBackend.removeNonExistingReadOnlyTenants(userTenantInfo);
        allTenants = this.searchGuardBackend.convertUserTenantsToRecord(userTenantInfo.data.tenants);
      } catch (error) {
        this.logger.info(`Could not retrieve the user tenants`);
      }
      if (!userTenantInfo || Object.keys(allTenants).length === 0) {
        throw new _missing_tenant_error.default('No tenant available for this user, please contact your system administrator.');
      }
    }

    // If we used any additional auth headers when authenticating, we need to store them in the session
    /* @todo Was used with sg_impersonate_as
    authResponse.session.additionalAuthHeaders = null;
    if (Object.keys(additionalAuthHeaders).length) {
      authResponse.session.additionalAuthHeaders = additionalAuthHeaders;
    }
      */

    const cookie = (await this.sessionStorageFactory.asScoped(request).get()) || {};
    authResponse.session = {
      ...cookie,
      ...authResponse.session
    };
    await this.sessionStorageFactory.asScoped(request).set(authResponse.session);
    return authResponse;
  }
  async logout({
    request,
    response
  }) {
    await this.clear(request, true);
    return response.ok({
      body: {
        authType: this.type,
        // @todo Use the kibana_url from the config?
        redirectURL: this.basePath + '/searchguard/login?type=' + this.type + 'Logout'
      }
    });
  }

  /**
   * Remove the credentials from the session cookie
   */
  async clear(request, explicitlyRemoveAuthType = false) {
    const sessionCookie = (await this.sessionStorageFactory.asScoped(request).get()) || {};
    const authHeaders = this.getAuthHeader(sessionCookie);
    // @todo Consider refactoring anything auth related into an "auth" property.
    delete sessionCookie.credentials;
    delete sessionCookie.username;
    if (explicitlyRemoveAuthType) {
      delete sessionCookie.authType;
      delete sessionCookie.authTypeId;
    }
    delete sessionCookie.additionalAuthHeaders;

    // Only try to delete the session if we really have auth headers
    if (authHeaders) {
      try {
        await this.searchGuardBackend.logoutSession(authHeaders);
      } catch (error) {
        this.logger.error(`Failed to delete the session token: ${error.stack}`);
      }
    }
    return await this.sessionStorageFactory.asScoped(request).set(sessionCookie);
  }
}
exports.default = AuthType;
module.exports = exports.default;
//# sourceMappingURL=data:application/json;charset=utf-8;base64,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