"use strict";

var _interopRequireDefault = require("@babel/runtime/helpers/interopRequireDefault");
Object.defineProperty(exports, "__esModule", {
  value: true
});
exports.MultitenancyLifecycle = void 0;
var _defineProperty2 = _interopRequireDefault(require("@babel/runtime/helpers/defineProperty"));
var _url = require("url");
var _lodash = require("lodash");
var _coreHttpRouterServerInternal = require("@kbn/core-http-router-server-internal");
var _multitenancy = require("../../../common/multitenancy");
/*
 *    Copyright 2020 floragunn GmbH
 *
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 * http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 */

class MultitenancyLifecycle {
  constructor({
    authManager,
    kerberos,
    searchGuardBackend,
    configService,
    sessionStorageFactory,
    logger,
    pluginDependencies,
    spacesService,
    kibanaCore,
    clusterClient
  }) {
    (0, _defineProperty2.default)(this, "onPreAuth", async (request, response, toolkit) => {
      const authType = this.configService.get('searchguard.auth.type');
      const debugEnabled = this.configService.get('searchguard.multitenancy.debug');
      const externalTenant = this.getExternalTenant(request, debugEnabled);

      // If we have an externalTenant, we will continue so that we can
      // update the cookie's tenant value
      if (!this.isRelevantPath(request) && !externalTenant) {
        return toolkit.next();
      }
      let selectedTenant = null;
      const {
        authHeaders,
        sessionCookie
      } = await this.getSession(request);
      const isAuthenticatedRequest = authType === 'proxy' || authHeaders && authHeaders.authorization ? true : false;

      // We may run into ugly issues with the capabilities endpoint here if
      // we let through an unauthenticated request, despite try/catch
      // Hence, only call the tenant endpoint if we are using proxy
      // or have an authorization header.
      if (!isAuthenticatedRequest) {
        return toolkit.next();
      }

      // Check if MT is enabled in the backend
      const {
        kibana_mt_enabled
      } = await this.searchGuardBackend.getKibanaInfoWithInternalUser();
      this.configService.set('searchguard.multitenancy.enabled', kibana_mt_enabled || false);

      // Skip early if MT is not enabled
      if (!kibana_mt_enabled) {
        return toolkit.next();
      }

      // As of 9.2 we can't access the clusterClient's config anymore.
      // Check the comment in serverPlugin.js for how to retrieve the
      // config right now. However, in multitenancy.js we throw an
      // error if sgtenant is not included in the list, so this
      // most likely did not have any real effect.
      // This may be subject to change:
      // https://github.com/elastic/kibana/issues/119862
      /*
      try {
        if (this.clusterClient.config.requestHeadersWhitelist.indexOf('sgtenant') === -1) {
          this.clusterClient.config.requestHeadersWhitelist.push('sgtenant');
        }
      } catch (error) {
        this.logger.error(`Multitenancy: Could not check headers whitelist ${request.url.pathname}. ${error}`);
      }
       */

      // The capabilities route may break the entire screen if
      // we get a 401 when retrieving the tenants. So for the
      // default capabilities, we can just skip MT here.
      if (request.url.pathname.indexOf('capabilities') > -1 && request.url.searchParams.get('useDefaultCapabilities') === "true") {
        return toolkit.next();
      }
      let userTenantInfo;
      try {
        // We need the user's data from the backend to validate the selected tenant
        userTenantInfo = await this.searchGuardBackend.getUserTenantInfo(authHeaders);
        if (!userTenantInfo.data.multi_tenancy_enabled) {
          // MT is disabled, we don't need to do anything.
          // This should have been checked earlier though, so this is just a fail safe.
          return toolkit.next();
        }
        selectedTenant = await this.getSelectedTenant({
          request,
          sessionCookie,
          username: userTenantInfo.data.username,
          externalTenant,
          userTenantInfo
        });
      } catch (error) {
        this.logger.error(`Multitenancy: Could not get tenant info from ${request.url.pathname}. ${error}`);
        if (error.statusCode === 401) {
          return toolkit.next();
        }
      }
      const requestHasRequestedTenant = externalTenant || typeof sessionCookie.tenant !== 'undefined';
      // If we have an external tenant, but the selectedTenant is null
      // after validation, that means that the user does not have
      // access to the requested tenant, or it does not exist
      if (selectedTenant === null && requestHasRequestedTenant) {
        if (request.url.pathname.startsWith('/app') || request.url.pathname === '/') {
          // If we have the wrong tenant in the cookie, we need to reset the cookie tenant value
          const shouldResetCookieTenant = !externalTenant && typeof sessionCookie.tenant !== 'undefined';
          if (shouldResetCookieTenant) {
            delete sessionCookie.tenant;
            await this.sessionStorageFactory.asScoped(request).set(sessionCookie);
          }
          if (request.url.searchParams.get('sgtenantsmenu') !== _multitenancy.MISSING_TENANT_PARAMETER_VALUE) {
            return response.redirected({
              body: 'Wrong tenant',
              //statusCode: 401,
              headers: {
                'location': this.basePath + `/app/home?sgtenantsmenu=` + _multitenancy.MISSING_TENANT_PARAMETER_VALUE
              }
            });
          }
        } else {
          // TODO maybe just pass through and let the backend return an error?
          return response.unauthorized();
        }
      }

      // We have a tenant to use
      if (selectedTenant !== null) {
        const rawRequest = (0, _coreHttpRouterServerInternal.ensureRawRequest)(request);
        (0, _lodash.assign)(rawRequest.headers, authHeaders, {
          sgtenant: selectedTenant
        });
        if (this.pluginDependencies.spaces) {
          // If we have a new tenant with no default space, we need to create it.
          await this.spacesService.createDefaultSpace({
            request,
            selectedTenant
          });
        }
        if (selectedTenant !== sessionCookie.tenant) {
          // save validated tenant in the cookie
          sessionCookie.tenant = selectedTenant;
          await this.sessionStorageFactory.asScoped(request).set(sessionCookie);
        }
      } else {
        let authRequiredForRoute = false;
        try {
          authRequiredForRoute = request.route.options.authRequired;
        } catch (error) {
          // Ignore
        }

        // Could also be "optional" or false
        if (authRequiredForRoute === true) {
          return response.redirected({
            body: 'Missing Tenant',
            //statusCode: 401,
            headers: {
              'location': this.basePath + `/searchguard/login?err=` + _multitenancy.MISSING_TENANT_PARAMETER_VALUE
            }
          });
        }
      }
      return toolkit.next();
    });
    /**
     * Get and validate the selected tenant.
     * @param request
     * @param sessionCookie
     * @param {string} username
     * @param {string|null} externalTenant
     * @param userTenantInfo
     * @returns {Promise<string|null>}
     */
    (0, _defineProperty2.default)(this, "getSelectedTenant", async ({
      request,
      sessionCookie,
      username,
      externalTenant,
      userTenantInfo
    }) => {
      const debugEnabled = this.configService.get('searchguard.multitenancy.debug');
      const backend = this.searchGuardBackend;

      // default is the tenant stored in the tenants cookie
      let selectedTenant = sessionCookie && typeof sessionCookie.tenant !== 'undefined' ? sessionCookie.tenant : null;
      if (debugEnabled) {
        this.logger.info(`Multitenancy: tenant_storagecookie ${selectedTenant}`);
      }
      if (externalTenant) {
        selectedTenant = externalTenant;
      }

      /**
       * @type {Record<string, boolean>}
       */
      const userTenants = this.searchGuardBackend.convertUserTenantsToRecord(userTenantInfo.data.tenants);

      // if we have a tenant, check validity and set it
      if (typeof selectedTenant !== 'undefined' && selectedTenant !== null && selectedTenant !== "") {
        selectedTenant = backend.validateRequestedTenant(username, selectedTenant, userTenants);
      } else if (userTenantInfo && userTenantInfo.data.default_tenant) {
        selectedTenant = userTenantInfo.data.default_tenant;
      }
      if (debugEnabled) {
        this.logger.info(`Multitenancy: tenant_assigned ${selectedTenant}`);
      }
      return selectedTenant;
    });
    /**
     * Get the auth information needed to make user scoped requests
     * @param request
     * @returns {Promise<{sessionCookie: {}, authHeaders: *}>}
     */
    (0, _defineProperty2.default)(this, "getSession", async request => {
      let sessionCookie;
      let authHeaders = request.headers;
      if (this.authManager) {
        const authInstance = await this.authManager.getAuthInstanceByRequest({
          request
        });
        if (authInstance) {
          sessionCookie = await authInstance.getCookieWithCredentials(request);
          authHeaders = authInstance.getAuthHeader(sessionCookie);
        } else {
          sessionCookie = await this.sessionStorageFactory.asScoped(request).get();
        }
      } else if (this.kerberos) {
        sessionCookie = await this.kerberos.getCookieWithCredentials(request);
        authHeaders = this.kerberos.getAuthHeader(sessionCookie);
      } else {
        sessionCookie = await this.sessionStorageFactory.asScoped(request).get();
      }
      if (!sessionCookie) {
        sessionCookie = {};
      }
      return {
        sessionCookie,
        authHeaders
      };
    });
    (0, _defineProperty2.default)(this, "isRelevantPath", request => {
      const path = request.url.pathname;

      // MT is only relevant for these paths
      const relevantPaths = ['/internal', '/goto', '/opensearch', '/app', '/api', '/bootstrap.js'];

      // MT is not relevant in these patterns
      const ignorePatterns = ['/api/status', '/api/v1/auth/config', '/api/v1/auth/login', '/api/v1/systeminfo'];
      return path === '/' || relevantPaths.some(root => path.startsWith(root)) && !ignorePatterns.some(root => path.startsWith(root));
    });
    /**
     * Check if we have a tenant set as query parameter
     * or as header value
     *
     * @param request
     * @param debugEnabled
     * @returns {null|string}
     */
    (0, _defineProperty2.default)(this, "getExternalTenant", (request, debugEnabled = false) => {
      let externalTenant = null;
      // check for tenant information in HTTP header. E.g. when using the saved objects API
      if (request.headers.sgtenant || request.headers.sg_tenant) {
        externalTenant = request.headers.sgtenant ? request.headers.sgtenant : request.headers.sg_tenant;
        if (debugEnabled) {
          this.logger.info(`Multitenancy: tenant_http_header: ${externalTenant}`);
        }
      }
      // check for tenant information in GET parameter. E.g. when using a share link. Overwrites the HTTP header.
      if (request.url.searchParams.has('sg_tenant') || request.url.searchParams.has('sgtenant')) {
        externalTenant = request.url.searchParams.has('sg_tenant') ? request.url.searchParams.get('sg_tenant') : request.url.searchParams.get('sgtenant');
        if (debugEnabled) {
          this.logger.info(`Multitenancy: tenant_url_param' ${externalTenant}`);
        }
      }
      if (externalTenant !== null) {
        try {
          if (externalTenant.toLowerCase() === 'private') {
            return _multitenancy.PRIVATE_TENANT_NAME;
          }
          if (externalTenant.toLowerCase() === 'global' || externalTenant.toUpperCase() === _multitenancy.GLOBAL_TENANT_NAME) {
            return _multitenancy.GLOBAL_TENANT_NAME;
          }
        } catch (error) {
          this.logger.error(`Could not translate private/global tenant: ` + externalTenant);
        }
      }
      return externalTenant;
    });
    this.authManager = authManager;
    this.searchGuardBackend = searchGuardBackend;
    this.configService = configService;
    this.sessionStorageFactory = sessionStorageFactory;
    this.logger = logger;
    this.pluginDependencies = pluginDependencies;
    this.spacesService = spacesService;
    this.kerberos = kerberos;
    this.kibanaCore = kibanaCore;
    this.clusterClient = clusterClient;
    this.basePath = kibanaCore.http.basePath.get();
  }
}
exports.MultitenancyLifecycle = MultitenancyLifecycle;
//# sourceMappingURL=data:application/json;charset=utf-8;base64,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