"use strict";

Object.defineProperty(exports, "__esModule", {
  value: true
});
exports.ReadOnlyMode = void 0;
var _url = require("url");
var _lodash = require("lodash");
/* eslint-disable @kbn/eslint/require-license-header */

class ReadOnlyMode {
  constructor(logger) {
    this.logger = logger;
    this.readOnlyModeRoles = null;
    this.searchGuardBackend = null;
    this.configService = null;
  }
  async setupSync({
    kibanaCoreSetup,
    searchGuardBackend,
    configService
  }) {
    this.readOnlyModeRoles = configService.get('searchguard.readonly_mode.roles');
    this.searchGuardBackend = searchGuardBackend;
    this.configService = configService;
    this.kibanaCoreSetup = kibanaCoreSetup;
  }
  isAnonymousPage(request) {
    if (request.headers && request.headers.referer) {
      try {
        const {
          pathname
        } = (0, _url.parse)(request.headers.referer);
        const pathsToIgnore = ['login', 'logout', 'customerror'];
        if (pathsToIgnore.indexOf(pathname.split('/').pop()) > -1) {
          return true;
        }
      } catch (error) {
        this.logger.error(`Could not parse the referer for the capabilites: ${error.stack}`);
      }
    }
    return false;
  }
  async switcherHandler(request, uiCapabilities) {
    // Only change capabilities if relevant
    const isMTEnabled = this.configService.get('searchguard.multitenancy.enabled');
    if (this.readOnlyModeRoles.length === 0 && !isMTEnabled) {
      return uiCapabilities;
    }

    // Ignore for non authenticated paths
    if (this.isAnonymousPage(request)) {
      return uiCapabilities;
    }
    try {
      // TODO concurrent calls
      const authInfo = await this.searchGuardBackend.authinfo(request.headers);
      /**
       * @type {Record<string, boolean>}
       */
      let userTenants = authInfo.sg_tenants;
      if (this.hasReadOnlyRole(authInfo, this.readOnlyModeRoles)) {
        // A read only role trumps the tenant access rights
        return this.toggleForReadOnlyRole(uiCapabilities);
      } else if (isMTEnabled && this.isReadOnlyTenant(authInfo, userTenants)) {
        return this.toggleForReadOnlyTenant(uiCapabilities, this.configService);
      }
    } catch (error) {
      this.logger.error(`Could not check auth info: ${error.stack}`);
    }
    return uiCapabilities;
  }
  hasReadOnlyRole(authInfo, readOnlyModeRoles) {
    return authInfo.sg_roles.some(role => readOnlyModeRoles.includes(role));
  }

  /**
   * Check if current tenant is read only for the user
   * @param authInfo
   * @returns {boolean}
   */
  isReadOnlyTenant(authInfo, userTenants) {
    // The global tenant would be '' == falsey
    const currentTenant = authInfo.user_requested_tenant;
    if (currentTenant === '__user__') {
      // We don't limit the private tenant
      return false;
    }
    const isReadOnlyTenant = userTenants[currentTenant] !== true ? true : false;
    return isReadOnlyTenant;
  }
  toggleForReadOnlyRole(uiCapabilities) {
    const whitelist = ['home', 'dashboard', 'dashboards'];
    Object.keys(uiCapabilities).forEach(capability => {
      if (capability === 'navLinks') {
        // Hide navLinks
        Object.keys(uiCapabilities.navLinks).forEach(navLinkId => {
          uiCapabilities.navLinks[navLinkId] = whitelist.indexOf(navLinkId) > -1;
        });
      } else if (capability === 'catalogue') {
        // Hide features from the catalogue
        Object.keys(uiCapabilities.catalogue).forEach(appId => {
          uiCapabilities.catalogue[appId] = whitelist.indexOf(appId) > -1;
        });
      } else if (
      // Here we're looking for a show property
      typeof uiCapabilities[capability] === 'object' && typeof uiCapabilities[capability].show !== 'undefined') {
        // If an app has show = false, Kibana will redirect away from its url.
        uiCapabilities[capability].show = whitelist.indexOf(capability) > -1;
      }
    });
    const defaultUICapabilities = {
      dashboard: {
        createNew: false,
        showWriteControls: false,
        saveQuery: false
      }
    };
    const finalCapabilities = (0, _lodash.merge)(uiCapabilities, defaultUICapabilities);
    return finalCapabilities;
  }
  toggleForReadOnlyTenant(uiCapabilities) {
    const defaultTenantOnlyCapabilities = {
      navLinks: {
        'kibana:stack_management': false,
        management: false
      },
      catalogue: {
        advanced_settings: false,
        index_patterns: false
      },
      dashboard: {
        createNew: false,
        showWriteControls: false,
        saveQuery: false
      },
      visualize: {
        createShortUrl: false,
        delete: false,
        save: false,
        saveQuery: false
      },
      management: {
        kibana: {
          indexPatterns: false
        }
      }
    };
    const finalCapabilities = (0, _lodash.merge)(uiCapabilities, defaultTenantOnlyCapabilities);
    return finalCapabilities;
  }
}
exports.ReadOnlyMode = ReadOnlyMode;
//# sourceMappingURL=data:application/json;charset=utf-8;base64,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