"use strict";

Object.defineProperty(exports, "__esModule", {
  value: true
});
exports.defineConfigurationRoutes = defineConfigurationRoutes;
exports.deleteConfigCache = deleteConfigCache;
exports.deleteConfigResource = deleteConfigResource;
exports.getConfigAliases = getConfigAliases;
exports.getConfigDataStreams = getConfigDataStreams;
exports.getConfigIndexMappings = getConfigIndexMappings;
exports.getConfigIndices = getConfigIndices;
exports.getConfigResourceById = getConfigResourceById;
exports.getConfigResourcesByType = getConfigResourcesByType;
exports.getRestApiInfo = getRestApiInfo;
exports.updateConfigResource = updateConfigResource;
var _configSchema = require("@kbn/config-schema");
var _wrap_elasticsearch_error = require("../../../../utils/wrap_elasticsearch_error");
var _constants = require("../../../../utils/constants");
/*
 *    Copyright 2020 floragunn GmbH
 *
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 * http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 */

function getConfigResourcesByType({
  searchGuardConfigurationBackend,
  logger
}) {
  return async function (context, request, response) {
    try {
      const results = await searchGuardConfigurationBackend.list(request.headers, request.params.resourceName);
      return response.ok({
        body: {
          total: Object.keys(results).length,
          data: results
        }
      });
    } catch (error) {
      logger.error(`getConfigResourcesByType: ${error.stack}`);
      return response.customError((0, _wrap_elasticsearch_error.wrapForCustomError)(error));
    }
  };
}
function getConfigResourceById({
  searchGuardConfigurationBackend,
  logger
}) {
  return async function (context, request, response) {
    try {
      const body = await searchGuardConfigurationBackend.get(request.headers, request.params.resourceName, request.params.id);
      return response.ok({
        body
      });
    } catch (error) {
      logger.error(`getConfigResourceById: ${error.stack}`);
      return response.customError((0, _wrap_elasticsearch_error.wrapForCustomError)(error));
    }
  };
}
function deleteConfigResource({
  searchGuardConfigurationBackend,
  logger
}) {
  return async function (context, request, response) {
    try {
      const result = await searchGuardConfigurationBackend.delete(request.headers, request.params.resourceName, request.params.id);
      return response.ok({
        body: {
          message: result.message
        }
      });
    } catch (error) {
      logger.error(`deleteConfigResource: ${error.stack}`);
      return response.customError((0, _wrap_elasticsearch_error.wrapForCustomError)(error));
    }
  };
}
function updateConfigResource({
  searchGuardConfigurationBackend,
  logger
}) {
  return async function (context, request, response) {
    try {
      const result = await searchGuardConfigurationBackend.save(request.headers, request.params.resourceName, request.params.id, request.body);
      return response.ok({
        body: {
          message: result.message
        }
      });
    } catch (error) {
      logger.error(`updateConfigResource: ${error.stack}`);
      return response.customError((0, _wrap_elasticsearch_error.wrapForCustomError)(error));
    }
  };
}
function deleteConfigCache({
  searchGuardConfigurationBackend,
  logger
}) {
  return async function (context, request, response) {
    try {
      const result = await searchGuardConfigurationBackend.clearCache(request.headers);
      return response.ok({
        body: {
          message: result.message
        }
      });
    } catch (error) {
      logger.error(`deleteConfigCache: ${error.stack}`);
      return response.customError((0, _wrap_elasticsearch_error.wrapForCustomError)(error));
    }
  };
}
function getRestApiInfo({
  searchGuardConfigurationBackend,
  logger
}) {
  return async function (context, request, response) {
    try {
      const body = await searchGuardConfigurationBackend.restapiinfo(request.headers);
      return response.ok({
        body
      });
    } catch (error) {
      logger.error(`getRestApiInfo: ${error.stack}`);
      return response.customError((0, _wrap_elasticsearch_error.wrapForCustomError)(error));
    }
  };
}
function getConfigIndices({
  searchGuardConfigurationBackend,
  logger
}) {
  return async function (context, request, response) {
    try {
      const {
        headers,
        body: {
          index
        }
      } = request;
      const body = await searchGuardConfigurationBackend.indices({
        headers,
        index
      });
      return response.ok({
        body
      });
    } catch (error) {
      logger.error(`getConfigIndices: ${error.stack}`);
      return response.customError((0, _wrap_elasticsearch_error.wrapForCustomError)(error));
    }
  };
}
function getConfigAliases({
  searchGuardConfigurationBackend,
  logger
}) {
  return async function (context, request, response) {
    try {
      const {
        headers,
        body: {
          alias
        }
      } = request;
      const body = await searchGuardConfigurationBackend.aliases({
        headers,
        alias
      });
      return response.ok({
        body
      });
    } catch (error) {
      logger.error(`getConfigAliases: ${error.stack}`);
      return response.customError((0, _wrap_elasticsearch_error.wrapForCustomError)(error));
    }
  };
}
function getConfigIndexMappings({
  searchGuardConfigurationBackend,
  logger
}) {
  return async function (context, request, response) {
    try {
      const body = await searchGuardConfigurationBackend.getIndexMappings({
        headers: request.headers,
        body: request.body
      });
      return response.ok({
        body
      });
    } catch (error) {
      logger.error(`getConfigIndexMappings: ${error.stack}`);
      return response.customError((0, _wrap_elasticsearch_error.wrapForCustomError)(error));
    }
  };
}
function getConfigDataStreams({
  searchGuardConfigurationBackend,
  logger
}) {
  return async function (context, request, response) {
    try {
      const body = await searchGuardConfigurationBackend.dataStreams({
        headers: request.headers,
        body: request.body
      });
      return response.ok({
        body
      });
    } catch (error) {
      logger.error(`getConfigDataStreams: ${error.stack}`);
      return response.customError((0, _wrap_elasticsearch_error.wrapForCustomError)(error));
    }
  };
}

/**
 * The backend API allows to manage the backend configuration.
 */
function defineConfigurationRoutes({
  searchGuardConfigurationBackend,
  kibanaCore,
  logger
}) {
  const router = kibanaCore.http.createRouter();

  /**
   * Returns a list of resource instances.
   *
   */
  router.get({
    path: `${_constants.API_ROOT}/configuration/{resourceName}`,
    validate: {
      params: _configSchema.schema.object({
        resourceName: _configSchema.schema.string()
      })
    },
    options: {
      authRequired: true
    }
  }, getConfigResourcesByType({
    searchGuardConfigurationBackend,
    logger
  }));

  /**
   * Returns a resource instance.
   *
   * Response sample:
   *
   * {
   *   "id": "kibiuser",
   * }
   */
  router.get({
    path: `${_constants.API_ROOT}/configuration/{resourceName}/{id}`,
    validate: {
      params: _configSchema.schema.object({
        resourceName: _configSchema.schema.string(),
        id: _configSchema.schema.string()
      })
    },
    options: {
      authRequired: true
    }
  }, getConfigResourceById({
    searchGuardConfigurationBackend,
    logger
  }));

  /**
   * Deletes a resource instance.
   *
   * Response sample:
   *
   * {
   *   "message": "Deleted user username"
   * }
   */
  router.delete({
    path: `${_constants.API_ROOT}/configuration/{resourceName}/{id}`,
    validate: {
      params: _configSchema.schema.object({
        resourceName: _configSchema.schema.string(),
        id: _configSchema.schema.string()
      })
    },
    options: {
      authRequired: true
    }
  }, deleteConfigResource({
    searchGuardConfigurationBackend,
    logger
  }));

  /**
   * Updates or creates a resource
   *
   * Request sample:
   *
   * {
   *   "password": "password"
   * }
   */
  router.post({
    path: `${_constants.API_ROOT}/configuration/{resourceName}/{id}`,
    validate: {
      params: _configSchema.schema.object({
        resourceName: _configSchema.schema.string(),
        id: _configSchema.schema.string()
      }),
      body: _configSchema.schema.object({}, {
        unknowns: 'allow'
      })
    },
    options: {
      authRequired: true
    }
  }, updateConfigResource({
    searchGuardConfigurationBackend,
    logger
  }));
  router.delete({
    path: `${_constants.API_ROOT}/configuration/cache`,
    validate: false,
    options: {
      authRequired: true
    }
  }, deleteConfigCache({
    searchGuardConfigurationBackend,
    logger
  }));
  router.get({
    path: `${_constants.API_ROOT}/restapiinfo`,
    validate: false,
    options: {
      authRequired: true
    }
  }, getRestApiInfo({
    searchGuardConfigurationBackend,
    logger
  }));
  router.post({
    path: `${_constants.API_ROOT}/configuration/indices`,
    validate: {
      body: _configSchema.schema.object({
        index: _configSchema.schema.oneOf([_configSchema.schema.string(), _configSchema.schema.arrayOf(_configSchema.schema.string())])
      })
    },
    options: {
      authRequired: true
    }
  }, getConfigIndices({
    searchGuardConfigurationBackend,
    logger
  }));
  router.post({
    path: `${_constants.API_ROOT}/configuration/aliases`,
    validate: {
      body: _configSchema.schema.object({
        alias: _configSchema.schema.oneOf([_configSchema.schema.string(), _configSchema.schema.arrayOf(_configSchema.schema.string())])
      })
    },
    options: {
      authRequired: true
    }
  }, getConfigAliases({
    searchGuardConfigurationBackend,
    logger
  }));
  router.post({
    path: `${_constants.API_ROOT}/configuration/index_mappings`,
    validate: {
      body: _configSchema.schema.object({
        index: _configSchema.schema.arrayOf(_configSchema.schema.string())
      })
    },
    options: {
      authRequired: true
    }
  }, getConfigIndexMappings({
    searchGuardConfigurationBackend,
    logger
  }));
  router.post({
    path: `${_constants.API_ROOT}/configuration/data_streams`,
    validate: {
      body: _configSchema.schema.object({
        dataStream: _configSchema.schema.oneOf([_configSchema.schema.string(), _configSchema.schema.arrayOf(_configSchema.schema.string())])
      })
    },
    options: {
      authRequired: true
    }
  }, getConfigDataStreams({
    searchGuardConfigurationBackend,
    logger
  }));
}
//# sourceMappingURL=data:application/json;charset=utf-8;base64,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